/*
 * @(#)ArrayType.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

/**
 * A class for handling array type information for symbols in the
 * MIBs.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class ArrayType extends Type {

    /**
     * The base type of the array.
     */
    private Type baseType;

    /**
     * The size constraint, or null.
     */
    private SizeConstraint size;

    /**
     * Creates a new array type with the given base type.
     *
     * @param  baseType     the array base type
     */
    public ArrayType(Type baseType) {
        this(baseType, null);
    }

    /**
     * Creates a new array type with the given base type and size.
     *
     * @param  baseType     the array base type
     * @param  size         a size constraint
     */
    public ArrayType(Type baseType, SizeConstraint size) {
        this.baseType = baseType;
        this.size = size;
    }

    /**
     * Checks if this type equals another.
     *
     * @param  obj       an object
     * @return true if the types are equal, false otherwise
     */
    public boolean equals(Object obj) {
        return this == obj;
    }

    /**
     * Returns a string description of this type.
     *
     * @return a string description of the type
     */
    public String toString() {
        if (size == null) {
            return "Array of " + baseType;
        } else {
            return "Array of [" + size + "]" + baseType;
        }
    }

    /**
     * Transfers the type information from this type to a type
     * converter object. The calling conventions declared in the
     * TypeConverter class are followed.
     *
     * @param   converter     a type converter
     */
    public void transferType(TypeConverter converter) {
        converter.createVector(baseType);
        if (size != null) {
            size.transferConstraint(converter);
        }
    }

}
