/*
 * @(#)SimpleMibPrinter.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.example;

import net.percederberg.mib.Mib;
import net.percederberg.mib.MibTreeModel;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.symbol.ValueSymbol;

/**
 * A program that prints the contents of a MIB file. This program has
 * been created as an example of how the MIB parser can be used.
 *
 * @author   Per Cederberg, per@percederberg.net
 * @version  1.0
 */
public class SimpleMibPrinter extends Object {

    /**
     * The application main entry point.
     *
     * @param args    the command-line parameters
     */
    public static void main(String[] args) {
        Mib           mib;
        MibTreeModel  tree;

        // Check command-line arguments
        if (args.length != 1) {
            printHelp("Wrong number of arguments");
            System.exit(1);
        }

        // Load MIB file and print tree
        try {
            mib = new Mib(args[0]);
            tree = new MibTreeModel(mib);
            System.err.print(mib.getAllMessages());
            printTree(tree);
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(2);
        }
    }

    /**
     * Prints command-line help information.
     *
     * @param error    an optional error message, or null
     */
    private static void printHelp(String error) {
        System.err.println("Syntax: java " + SimpleMibPrinter.class +
                           " <mibfile>");
        System.err.println();
        if (error != null) {
            System.err.print("Error: ");
            System.err.println(error);
            System.err.println();
        }
    }

    /**
     * Prints a tree model on standard output.
     *
     * @param tree     the tree model to print
     */
    private static void printTree(MibTreeModel tree) {
        printTreeNode(tree, tree.getRoot(), "");
    }

    /**
     * Prints a tree model node on standard output.
     *
     * @param tree     the tree model
     * @param node     the tree node to print
     * @param indent   the indentation string
     */
    private static void printTreeNode(MibTreeModel tree,
                                      Object node,
                                      String indent) {

        ValueSymbol  value;
        TypeSymbol   type;
        int          children;

        System.out.print(indent + node.toString());
        if (node instanceof ValueSymbol) {
            value = (ValueSymbol) node;
            System.out.print(", OID: " + value.getOID() +
                             ", Type: " + value.getType());
        } else if (node instanceof TypeSymbol) {
            type = (TypeSymbol) node;
            System.out.print(", Type: " + type.getType());
        }
        System.out.println();
        children = tree.getChildCount(node);
        for (int i = 0; i < children; i++) {
            printTreeNode(tree,
                          tree.getChild(node, i),
                          indent + "  ");
        }
    }
}
