/*
 * @(#)FirstPassAnalysis.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import net.percederberg.mib.asn1.analysis.DepthFirstAdapter;
import net.percederberg.mib.asn1.node.Node;
import net.percederberg.mib.symbol.Symbol;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.symbol.ValueSymbol;

/**
 * The first pass analysis of the parse tree. Creates symbols for
 * all identifiers in the MIB and adds them to the symbol table.
 * Also checks for unsupported syntactic constructions, unsupported
 * types and unknown import symbols.<p>
 *
 * During the first pass analysis the symbols will not be analyzed
 * for their type, though, as such an analysis requires full
 * knowledge of all type and value names used. That type of
 * analysis is performed in the second pass.<p>
 *
 * This class is used internally during MIB creation.
 *
 * @version  1.5
 * @author   Per Cederberg, per@percederberg.net
 */
class FirstPassAnalysis extends DepthFirstAdapter {

    /**
     * The MIB to work on.
     */
    private Mib mib = null;
    
    /**
     * The symbol factory.
     */
    private SymbolFactory factory = null;

    /**
     * Creates a new first pass analyzer.
     *
     * @param   mib       the MIB to add symbols to
     */
    public FirstPassAnalysis(Mib mib) {
        this.mib = mib;
        this.factory = new SymbolFactory(mib);
    }

    /**
     * Creates the default symbols that are always available.
     *
     * @param node    the parse tree node
     */
    protected void inStart(Node node) {
        mib.addSymbol(factory.createSymbol("iso"));
    }

    /**
     * Sets the output value to the name of the module reference.
     *
     * @param node    the parse tree node
     */
    protected void outModuleDefinition(Node node) {
        String   name;

        name = (String) getOut(node.childOfType(Node.MODULEIDENTIFIER));
        mib.setName(name);
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inTagDefault(Node node) {
        mib.addWarning("unsupported construct: ... TAGS",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Sets the output value to the name of the module reference.
     *
     * @param node    the parse tree node
     */
    protected void outModuleIdentifier(Node node) {
        setOut(node, getOut(node.childOfType(Node.MODULEREFERENCE)));
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inExports(Node node) {
        mib.addWarning("unsupported construct: EXPORTS ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Analyzes the symbol if inside imports.
     *
     * @param node    the parse tree node
     */
    protected void outSymbol(Node node) {
        String  id;
        Symbol  sym;

        if (node.isAncestor(Node.IMPORTS)
         && node.childrenOfType(Node.DEFINEDMACRONAME) == 0) {

            id = (String) getOut(node.childOfType(Node.IDENTIFIER));
            sym = factory.findSymbol(id);
            if (sym != null) {
                mib.addSymbol(sym);
            } else {
                mib.addWarning("import unknown: " + id);
            }
        }
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inMacroDefinition(Node node) {
        mib.addWarning("unsupported construct: MACRO ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Creates and adds a type symbol.
     *
     * @param node    the parse tree node
     */
    protected void outTypeAssignment(Node node) {
        String id = (String) getOut(node.childOfType(Node.TYPEREFERENCE));
        mib.addSymbol(new TypeSymbol(id));
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inBitStringType(Node node) {
        mib.addWarning("unsupported type: BIT STRING",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inChoiceType(Node node) {
        mib.addWarning("unsupported type: CHOICE ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inEnumeratedType(Node node) {
        mib.addWarning("unsupported type: ENUMERATED ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inSelectionType(Node node) {
        mib.addWarning("unsupported type: name < ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inTaggedType(Node node) {
        mib.addWarning("unsupported type: [X] ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inAnyType(Node node) {
        mib.addWarning("unsupported type: ANY",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates an warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inAlphabetConstraint(Node node) {
        mib.addWarning("unsupported constraint: FROM ...",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Creates and adds a value symbol.
     *
     * @param node    the parse tree node
     */
    protected void outValueAssignment(Node node) {
        String id = (String) getOut(node.childOfType(Node.IDENTIFIER));
        mib.addSymbol(new ValueSymbol(id));
    }

    /**
     * Sets the output value to the identifier string.
     *
     * @param node    the parse tree node
     */
    protected void outIdentifier(Node node) {
        setOut(node, getOut(node.childOfType(Node.TIDENT)));
    }

    /**
     * Sets the output value to the identifier string.
     *
     * @param node    the parse tree node
     */
    protected void outModuleReference(Node node) {
        setOut(node, getOut(node.childOfType(Node.TIDENT)));
    }

    /**
     * Sets the output value to the identifier string.
     *
     * @param node    the parse tree node
     */
    protected void outTypeReference(Node node) {
        setOut(node, getOut(node.childOfType(Node.TIDENT)));
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inSnmpObjectGroupMacroType(Node node) {
        mib.addWarning("unsupported type: OBJECT-GROUP",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inSnmpNotificationGroupMacroType(Node node) {
        mib.addWarning("unsupported type: NOTIFICATION-GROUP",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inSnmpModuleComplianceMacroType(Node node) {
        mib.addWarning("unsupported type: MODULE-COMPLIANCE",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inSnmpAgentCapabilitiesMacroType(Node node) {
        mib.addWarning("unsupported type: AGENT-CAPABILITIES",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inTBOOLEAN(Node node) {
        mib.addWarning("unsupported type: BOOLEAN",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inTREAL(Node node) {
        mib.addWarning("unsupported type: REAL",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Generates a warning if called.
     *
     * @param node    the parse tree node
     */
    protected void inTNULL(Node node) {
        mib.addWarning("unsupported type and value: NULL",
                       node.firstLine(),
                       node.lastLine());
    }

    /**
     * Sets the output value to the token image.
     *
     * @param node    the parse tree node
     */
    protected void outTIDENT(Node node) {
        setOut(node, node.toString());
    }
}
