/*
 * @(#)Node.java                  1.3              31 August 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.asn1.node;

import net.percederberg.mib.asn1.analysis.Switch;
import net.percederberg.mib.asn1.analysis.Switchable;

/**
 * This abstract class is the interface for all the nodes in the parse
 * tree.
 *
 * @version  1.3
 * @author   Per Cederberg, per@percederberg.net
 */
public abstract class Node implements NodeConstants, Switchable {

   /**
    * The kind of token. (Unfortunately this wasn't possible to
    * make private since the parser uses it...)
    */
   public int     kind = -1;

   /**
    * The parent node
    */
   private Node   parent = null;

   /**
    * Returns the node name from the node type constant. For undefined
    * nodes null will be returned.
    *
    * @param  nodeType    a node type constant value
    * @return a node name string
    */
   public static String toName(int nodeType) {
      try {
         if (nodeType < 100) {
            return Node.tokenNodeName[nodeType];
         } else {
            return Node.productionNodeName[nodeType - 100];
         }
      } catch (Exception e) {
         return null;
      }
   }

   /**
    * Returns the node name from the node object. For undefined
    * nodes null will be returned.
    *
    * @param  node    a node object
    * @return a node name string
    */
   public static String toName(Node node) {
      return toName(node.kind);
   }

   /**
    * Calls the relevant caseXXX method in the given switch.
    *
    * @param sw   a switch object
    */
   public void apply(Switch sw) {
      // This implementation is not intelligent, but impossible to avoid
      // as there are no subclasses for each type of node...
      switch (kind) {
      case EOF:
         sw.caseEOF(this);
         break;
      case TDOT:
         sw.caseTDOT(this);
         break;
      case TCOMMA:
         sw.caseTCOMMA(this);
         break;
      case TSEMI_COLON:
         sw.caseTSEMI_COLON(this);
         break;
      case TLEFTPAREN:
         sw.caseTLEFTPAREN(this);
         break;
      case TRIGHTPAREN:
         sw.caseTRIGHTPAREN(this);
         break;
      case TLEFTBRACE:
         sw.caseTLEFTBRACE(this);
         break;
      case TRIGHTBRACE:
         sw.caseTRIGHTBRACE(this);
         break;
      case TLEFTBRACKET:
         sw.caseTLEFTBRACKET(this);
         break;
      case TRIGHTBRACKET:
         sw.caseTRIGHTBRACKET(this);
         break;
      case TMINUS:
         sw.caseTMINUS(this);
         break;
      case TLESSTHAN:
         sw.caseTLESSTHAN(this);
         break;
      case TBAR:
         sw.caseTBAR(this);
         break;
      case TASSIGN:
         sw.caseTASSIGN(this);
         break;
      case TDEFINITIONS:
         sw.caseTDEFINITIONS(this);
         break;
      case TEXPLICIT:
         sw.caseTEXPLICIT(this);
         break;
      case TIMPLICIT:
         sw.caseTIMPLICIT(this);
         break;
      case TTAGS:
         sw.caseTTAGS(this);
         break;
      case TBEGIN:
         sw.caseTBEGIN(this);
         break;
      case TEND:
         sw.caseTEND(this);
         break;
      case TEXPORTS:
         sw.caseTEXPORTS(this);
         break;
      case TIMPORTS:
         sw.caseTIMPORTS(this);
         break;
      case TFROM:
         sw.caseTFROM(this);
         break;
      case TMACRO:
         sw.caseTMACRO(this);
         break;
      case TINTEGER:
         sw.caseTINTEGER(this);
         break;
      case TREAL:
         sw.caseTREAL(this);
         break;
      case TBOOLEAN:
         sw.caseTBOOLEAN(this);
         break;
      case TNULL:
         sw.caseTNULL(this);
         break;
      case TBIT:
         sw.caseTBIT(this);
         break;
      case TOCTET:
         sw.caseTOCTET(this);
         break;
      case TSTRING:
         sw.caseTSTRING(this);
         break;
      case TENUMERATED:
         sw.caseTENUMERATED(this);
         break;
      case TSEQUENCE:
         sw.caseTSEQUENCE(this);
         break;
      case TSET:
         sw.caseTSET(this);
         break;
      case TOF:
         sw.caseTOF(this);
         break;
      case TCHOICE:
         sw.caseTCHOICE(this);
         break;
      case TUNIVERSAL:
         sw.caseTUNIVERSAL(this);
         break;
      case TAPPLICATION:
         sw.caseTAPPLICATION(this);
         break;
      case TPRIVATE:
         sw.caseTPRIVATE(this);
         break;
      case TANY:
         sw.caseTANY(this);
         break;
      case TDEFINED:
         sw.caseTDEFINED(this);
         break;
      case TBY:
         sw.caseTBY(this);
         break;
      case TOBJECT:
         sw.caseTOBJECT(this);
         break;
      case TIDENTIFIER:
         sw.caseTIDENTIFIER(this);
         break;
      case TINCLUDES:
         sw.caseTINCLUDES(this);
         break;
      case TMIN:
         sw.caseTMIN(this);
         break;
      case TMAX:
         sw.caseTMAX(this);
         break;
      case TSIZE:
         sw.caseTSIZE(this);
         break;
      case TWITH:
         sw.caseTWITH(this);
         break;
      case TCOMPONENT:
         sw.caseTCOMPONENT(this);
         break;
      case TCOMPONENTS:
         sw.caseTCOMPONENTS(this);
         break;
      case TPRESENT:
         sw.caseTPRESENT(this);
         break;
      case TABSENT:
         sw.caseTABSENT(this);
         break;
      case TOPTIONAL:
         sw.caseTOPTIONAL(this);
         break;
      case TDEFAULT:
         sw.caseTDEFAULT(this);
         break;
      case TTRUE:
         sw.caseTTRUE(this);
         break;
      case TFALSE:
         sw.caseTFALSE(this);
         break;
      case TPLUS_INFINITY:
         sw.caseTPLUS_INFINITY(this);
         break;
      case TMINUS_INFINITY:
         sw.caseTMINUS_INFINITY(this);
         break;
      case TOBJECT_TYPE:
         sw.caseTOBJECT_TYPE(this);
         break;
      case TTRAP_TYPE:
         sw.caseTTRAP_TYPE(this);
         break;
      case TSYNTAX:
         sw.caseTSYNTAX(this);
         break;
      case TACCESS:
         sw.caseTACCESS(this);
         break;
      case TSTATUS:
         sw.caseTSTATUS(this);
         break;
      case TDESCRIPTION:
         sw.caseTDESCRIPTION(this);
         break;
      case TREFERENCE:
         sw.caseTREFERENCE(this);
         break;
      case TINDEX:
         sw.caseTINDEX(this);
         break;
      case TDEFVAL:
         sw.caseTDEFVAL(this);
         break;
      case TENTERPRISE:
         sw.caseTENTERPRISE(this);
         break;
      case TVARIABLES:
         sw.caseTVARIABLES(this);
         break;
      case TBINSTRING:
         sw.caseTBINSTRING(this);
         break;
      case THEXSTRING:
         sw.caseTHEXSTRING(this);
         break;
      case TCSTRING:
         sw.caseTCSTRING(this);
         break;
      case TUCASEFIRST_IDENT:
         sw.caseTUCASEFIRST_IDENT(this);
         break;
      case TLCASEFIRST_IDENT:
         sw.caseTLCASEFIRST_IDENT(this);
         break;
      case TNUMBER:
         sw.caseTNUMBER(this);
         break;
      case START:
         sw.caseStart(this);
         break;
      case MODULEDEFINITION:
         sw.caseModuleDefinition(this);
         break;
      case TAGDEFAULT:
         sw.caseTagDefault(this);
         break;
      case MODULEIDENTIFIER:
         sw.caseModuleIdentifier(this);
         break;
      case MODULEBODY:
         sw.caseModuleBody(this);
         break;
      case EXPORTS:
         sw.caseExports(this);
         break;
      case IMPORTS:
         sw.caseImports(this);
         break;
      case SYMBOLSFROMMODULELIST:
         sw.caseSymbolsFromModuleList(this);
         break;
      case SYMBOLSFROMMODULE:
         sw.caseSymbolsFromModule(this);
         break;
      case SYMBOLLIST:
         sw.caseSymbolList(this);
         break;
      case SYMBOL:
         sw.caseSymbol(this);
         break;
      case ASSIGNMENTLIST:
         sw.caseAssignmentList(this);
         break;
      case ASSIGNMENT:
         sw.caseAssignment(this);
         break;
      case MACRODEFINITION:
         sw.caseMacroDefinition(this);
         break;
      case MACROBODY:
         sw.caseMacroBody(this);
         break;
      case SKIPTOEND:
         sw.caseSkipToEND(this);
         break;
      case MACROREFERENCE:
         sw.caseMacroReference(this);
         break;
      case TYPEASSIGNMENT:
         sw.caseTypeAssignment(this);
         break;
      case TYPE:
         sw.caseType(this);
         break;
      case BUILTINTYPE:
         sw.caseBuiltinType(this);
         break;
      case INTEGERTYPE:
         sw.caseIntegerType(this);
         break;
      case NAMEDNUMBERLIST:
         sw.caseNamedNumberList(this);
         break;
      case NAMEDNUMBER:
         sw.caseNamedNumber(this);
         break;
      case SIGNEDNUMBER:
         sw.caseSignedNumber(this);
         break;
      case STRINGTYPE:
         sw.caseStringType(this);
         break;
      case BITSTRINGTYPE:
         sw.caseBitStringType(this);
         break;
      case SEQUENCETYPE:
         sw.caseSequenceType(this);
         break;
      case SEQUENCEOFTYPE:
         sw.caseSequenceOfType(this);
         break;
      case SETTYPE:
         sw.caseSetType(this);
         break;
      case SETOFTYPE:
         sw.caseSetOfType(this);
         break;
      case ELEMENTTYPELIST:
         sw.caseElementTypeList(this);
         break;
      case ELEMENTTYPE:
         sw.caseElementType(this);
         break;
      case NAMEDTYPE:
         sw.caseNamedType(this);
         break;
      case CHOICETYPE:
         sw.caseChoiceType(this);
         break;
      case ENUMERATEDTYPE:
         sw.caseEnumeratedType(this);
         break;
      case SELECTIONTYPE:
         sw.caseSelectionType(this);
         break;
      case TAGGEDTYPE:
         sw.caseTaggedType(this);
         break;
      case TAG:
         sw.caseTag(this);
         break;
      case CLASSNUMBER:
         sw.caseClassNumber(this);
         break;
      case CLASS:
         sw.caseClass(this);
         break;
      case ANYTYPE:
         sw.caseAnyType(this);
         break;
      case DEFINEDTYPE:
         sw.caseDefinedType(this);
         break;
      case CONSTRAINTLIST:
         sw.caseConstraintList(this);
         break;
      case CONSTRAINT:
         sw.caseConstraint(this);
         break;
      case VALUECONSTRAINT:
         sw.caseValueConstraint(this);
         break;
      case VALUERANGE:
         sw.caseValueRange(this);
         break;
      case LOWERENDPOINT:
         sw.caseLowerEndPoint(this);
         break;
      case UPPERENDPOINT:
         sw.caseUpperEndPoint(this);
         break;
      case SIZECONSTRAINT:
         sw.caseSizeConstraint(this);
         break;
      case ALPHABETCONSTRAINT:
         sw.caseAlphabetConstraint(this);
         break;
      case VALUEASSIGNMENT:
         sw.caseValueAssignment(this);
         break;
      case VALUE:
         sw.caseValue(this);
         break;
      case DEFINEDVALUE:
         sw.caseDefinedValue(this);
         break;
      case BUILTINVALUE:
         sw.caseBuiltinValue(this);
         break;
      case BOOLEANVALUE:
         sw.caseBooleanValue(this);
         break;
      case SPECIALREALVALUE:
         sw.caseSpecialRealValue(this);
         break;
      case NULLVALUE:
         sw.caseNullValue(this);
         break;
      case NAMEDVALUE:
         sw.caseNamedValue(this);
         break;
      case OBJECTIDENTIFIERVALUE:
         sw.caseObjectIdentifierValue(this);
         break;
      case OBJIDCOMPONENTLIST:
         sw.caseObjIdComponentList(this);
         break;
      case OBJIDCOMPONENT:
         sw.caseObjIdComponent(this);
         break;
      case NAMEANDNUMBERFORM:
         sw.caseNameAndNumberForm(this);
         break;
      case BINARYSTRING:
         sw.caseBinaryString(this);
         break;
      case HEXSTRING:
         sw.caseHexString(this);
         break;
      case CHARSTRING:
         sw.caseCharString(this);
         break;
      case NUMBER:
         sw.caseNumber(this);
         break;
      case IDENTIFIER:
         sw.caseIdentifier(this);
         break;
      case MODULEREFERENCE:
         sw.caseModuleReference(this);
         break;
      case TYPEREFERENCE:
         sw.caseTypeReference(this);
         break;
      case DEFINEDMACROTYPE:
         sw.caseDefinedMacroType(this);
         break;
      case DEFINEDMACRONAME:
         sw.caseDefinedMacroName(this);
         break;
      case SNMPOBJECTTYPEMACROTYPE:
         sw.caseSnmpObjectTypeMacroType(this);
         break;
      case SNMPTRAPTYPEMACROTYPE:
         sw.caseSnmpTrapTypeMacroType(this);
         break;
      case SNMPACCESSPART:
         sw.caseSnmpAccessPart(this);
         break;
      case SNMPSTATUSPART:
         sw.caseSnmpStatusPart(this);
         break;
      case SNMPDESCRPART:
         sw.caseSnmpDescrPart(this);
         break;
      case SNMPREFERPART:
         sw.caseSnmpReferPart(this);
         break;
      case SNMPINDEXPART:
         sw.caseSnmpIndexPart(this);
         break;
      case TYPEORVALUELIST:
         sw.caseTypeOrValueList(this);
         break;
      case TYPEORVALUE:
         sw.caseTypeOrValue(this);
         break;
      case SNMPDEFVALPART:
         sw.caseSnmpDefValPart(this);
         break;
      case SNMPVARPART:
         sw.caseSnmpVarPart(this);
         break;
      case VARTYPES:
         sw.caseVarTypes(this);
         break;
      default:
         // Nothing done for unknown nodes
      }
   }

   /**
    * Returns the child coming after the given child.
    *
    * @param  child     a child node
    * @return the next child, or null if no next child
    */
   public Node childAfter(Node child) {
      return null;
   }

   /**
    * Returns a child with the given index.
    *
    * @param  index   a child index, from 0 to children()-1
    * @return a child node, or null
    */
   public Node childAt(int index) {
      return null;
   }

   /**
    * Returns the first child of the given type.
    *
    * @param  type    a node type constant value
    * @return a child node, or null
    */
   public Node childOfType(int type) {
      return null;
   }

   /**
    * Returns the number of children to the current node.
    *
    * @return  the number of children, or zero for no children
    */
   public int children() {
      return 0;
   }

   /**
    * Returns the number of children of a given type to the current node.
    *
    * @param type     the type number
    *
    * @return the number of children, or zero for no children
    */
   public int childrenOfType(int type) {
      return 0;
   }

   /**
    * Returns the parent node of this node.
    *
    * @return the parent node, or null for no parent
    */
   public Node getParent() {
      return parent;
   }

   /**
    * Checks if this node has an ancestor of the given type.
    *
    * @param  type      a node type constant value
    * @return true if some ancestor has the given type, false otherwise
    */
   public boolean isAncestor(int type) {
      Node  node = this;

      while (node != null) {
         if (node.isParent(type)) {
            return true;
         }
         node = node.getParent();
      }
      return false;
   }

   /**
    * Checks if this node has a parent of the given type.
    *
    * @param  type      a node type constant value
    * @return true if the parent has the given type, false otherwise
    */
   public boolean isParent(int type) {
      if (parent != null) {
         return this.parent.kind == type;
      } else {
         return false;
      }
   }

   /**
    * Checks if this node is of the given type.
    *
    * @param  type     a node type constant value
    * @return true if same type, otherwise false
    */
   public boolean isType(int type) {
      return type == this.kind;
   }

   /**
    * Sets the given node as parent node to this node.
    *
    * @param  parent     the node parent
    */
   public void setParent(Node parent) {
      this.parent = parent;
   }

   /**
    * Finds the first line for the node.
    *
    * @return   the first line, or -1 if no token present
    */
   public int firstLine() {
      return -1;
   }

   /**
    * Finds the first column for the node.
    *
    * @return   the first column, or -1 if no token present
    */
   public int firstColumn() {
      return -1;
   }

   /**
    * Finds the last line for the node.
    *
    * @return   the last line, or -1 if no token present
    */
   public int lastLine() {
      return -1;
   }

   /**
    * Finds the last column for the node.
    *
    * @return   the last column, or -1 if no token present
    */
   public int lastColumn() {
      return -1;
   }

   /**
    * Returns a string representation of this node. Possible children
    * will not be contained in this string. This method should be
    * overridden by subclasses.
    *
    * @return  a string representation of this node
    */
   public String toString() {
      if (toName(kind) != null) {
         return toName(kind);
      } else {
         return "*** UNKNOWN NODE ***";
      }
   }
}

