#! /usr/bin/env python

""" -------------------------------------------------------------------------------
  Constructs an OPenOffice text file from a pickled file made by TexConv.py

                        written by Nick Thomas

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  ---------------------------------------------------------------------------

  The handling sequence is table-driven by objDict{} and specFunc{}
    examine the latter to navigate and find the functions doing the work
  objDict{} is obtained from a pickled file made by TexConv.py: see that program
        for its structure
  object indices are extracted in sequence from this by gv.extractObj()
        and handled by the functions listed in specFunc{}
  this approach allows the handling sequence to impose special sequences to suit
        the structure of OpenOffice xml files, especially table and font handling
  processing starts at comment # STARTS HERE
  major objects are handled by calling handleObject() and then handleStart()
  subordinate objects are handled by calling handleObject()
  handleObject() and handleStart() look up the required handling function in
        specFunc{}

  Font styles and sizes are handled by special functions using the dictionaries;
  dictionary entries for Text have prefixes such as AC to enable the correct
  handling function to be called.

  Very long lines are left to the word processor to handle, except URLs

  The following graphics formats can be correctly sized:
        bmp
        eps
        gif
        jpg and jpeg
        pcx
        png
        ps
        xbm
      other types are sized as 12 x 9 cm
"""
"       BUT NOTE: OpenOffice apparently cannot display PS or EPS images'"
#-----------------------------------------------------------------------------------

"""
    The platform-dependent variables read in from the configuration file ConvTexPaths,
    which should not need to be changed, are:
        Tmp     the path for temporary files
                e.g. /tmp               for Linux
                \WINDOWS\Temp           for windows
        ErrPath the path for the error-reporting file
                e.g. /tmp/TexError      for Linux
                \WINDOWS\Temp\TexError  for Windows
        docDir  the default directory where the XML files are constructed
                e.g. /tmp/officeXML/    for Linux
                \WINDOWS\Temp\officeXML\ for Windows
        pythonTmp       not used in this program
        dicPath path for dictionaries
                e.g. Documents/ConvLaTex/Dictionaries/          for Linux
                My Documents\ConvLaTex\Dictionaries\            for Windows
        dirySep directory path separator
                /       for Linux
                \       for Windows
        xmlSep  separator used within XML documents (probably redundant)
                / for all platforms so far
        urlSep  separator for URLs (probably redundant)
                / for all platforms so far
"""
#-----------------------------------------------------------------------------------

""" EXCEPTIONS FROM LyX/Tex --------------------------------------------------------

    Equation labels
        all equation labels, including equation array labels, are simple numbers
        in brackets i.e. without reference to the section etc.

    Equation array labels
        only those equations in an equation array which possess a cross-reference
        are labelled, as other labels would obviously be redundant;
        this avoids unnecessary "fussiness"

    Sideways text in table cells
        OpenOffice cannot handle this on a cell-by-cell basis, but only for all header
        row cells and/or all other row cells.  The feature is only implemented here
        for header row cells, and then applies to them all, otherwise it is ignored.
        The first row of a table is only defined as a header row if the sideways
        feature is required, to avoid other tables being "infected".

    Some maths fonts are not implemented as StarMath does not handle them:
        blackboard, calligraphic and fraktur

    Figure 'float'      figure & caption are put in a paragraph-anchored frame
    Table 'float'       table & caption are put in a paragraph-anchored frame

    Page Refs are left as section (or whatever) refs for now as paging has yet to
        be handled

------------------------------------------------------------------------------------
List of global functions
  not in dictionary specFunc:

        addLabels(labels)               add labels to equation array
        arraySize(s,i,Scale)            calculate size of an array
        bmpSize(pn)                     determine size of bmp picture
        calcCharWid()                   calculate character widths
        calcEqnSize()                   calculate size of an equation
        calcES(s,i,Scale)               recursive part of above
        calcPicSize(picName)            determine size of a picture
        calcTableWid()                  calculate table width in cm
        checkSyntax()                   check syntax of maths expression
        convDoc()                       main procedure
        copyMimetype()                  make mimetype file
        doAccent(style)                 handle accented chars from ISO 8859 2 - 4
        DOcentre(L)                     set gobal centre justification
        DOcline(L)                      handle cline (not understood)
        DOendCell(L)                    enter completed table cell
        DOhfill(L)                      enter horizontal fill
        DOhline(L)                      enter horizontal line in table
        DOleftAlign(L)                  set left-justify
        DOnoindent(L)                   remove all indentation
        DOnonumber(L)                   handle format item nonumber (ignored)
        DOrightAlign(L)                 set right-justify
        DOtabularnewline(L)             handle table end of line
        enterContents(diry,centred)     enter equation in main contents
        epsSize(pn)                     determine size of eps picture
        finishEqn(needsStyleName)       finish processing an equation
        finishTable()                   finish processing a table
        getCharWid(s)                   get width of char in cm x 1000
        getInt(s)                       convert s to binary (s[0] most significant)
        getVal(s)                       convert s to binary (s[0] least significant)
        gifSize(pn)                     determine size of gif picture
        initCell()                      initialise new table cell
        initTable()                     initialise new table
        jpgSize(pn)                     determine size of jpeg picture
        make_xRef()                     make reversed cross ref from specFunc
        makeCellStyle(borders,column)   make table cell style
        makeContentEqn(diry)            make content.xml file for an equation
        makeContentMain()               make main content.xml file
        makeDocDir()                    make directory in /tmp for document
        makeErrorFile(msg)              make error-report file in /tmp
        makeManifest()                  make manifest.xml file
        makeMeta()                      make meta.xml file
        makeSettingsEqn(diry)           make settings.xml file for an equation
        makeSettingsMain()              make main settings.xml file
        makeStyles()                    make styles.xml file
        parseInt(s)                     parse ascii integer from s
        parseInteger(s)                 ditto but only returns value, not tuple
        parseMathsTerm(s,i)             parse maths term from s[i:]
        parseParam(s)                   parse next table cell parameter
        pcxSize(pn)                     determine size of pcx picture
        pngSize(pn)                     determine size of png picture
        prefixHSpace(n)                 prefix th.text with n hard spaces
        psSize(pn)                      determine size of postscript picture
        r(x)                            convert x to ascii, rounded to 3 dec places
        readAsciiDict(dict)             read an ascii file into a dictionary
        readDict(dict)                  read a dictionary-formatted file
        removeErrFile()                 remove error-report file in /tmp
        reportErr()                     report error on terminal from error file
        restoreContext()                restore current context
        saveContext(major)              save current context
        setMulti(cf)                    replace default cell style with multiformat one
        sizeText(s)                     size a mixed term for equation
        textLength(s)                   calculate length of string s in cm*1000
        writeTableRow()                 write completed table row
        xbmSize(pn)                     determine size of xbm picture
"""

import  getopt  #for command-line param parsing
import  sys     #for above and general fns like exit()
import  os      #for operating system functions like remove()
import  pickle
import  datetime
import  operator
#import rgbimg          # for finding image size; DOES NOT WORK
from Tkinter import *   #GUI support module
import Tix
from Tix import *       #must do this as well as above import (!!)
import  zipfile
import  zlib
from os.path import join

# ---------------------------------------------------------------------------------

"Configuration presets are read in from file ConvOfficePresets, located in same"
"directory as the program"
"At present this contains the dictionary and xml paths, and printer setup string"

"Path presets are fixed for the platform, and are read in to enable the same"
"program to be used for different platforms"

# NUMERICAL PRESETS ----------------------------------------------------------------

rowGap=120              # gap between matrix rows
colGap=250              # column gap for matrices
picScale=45             # pix/picScale = cm for calculating Figure size was 37.8
eqnTextF=1.1            # factor for text in equations

#----------------------------------------------------------------------------------

fnr='File name required'
dpr='Dictionary Path required'
noOpen="Could not open file: "
picDir=''
xmlDir=''
isTerminal=0
major=0
textPara='<text:p text:style-name="'
textSpan='<text:span text:style-name="'
H=[0,0,'\\hardspace']
B=[0,0,'\\bullet']
UCletters='ABCDEFGHIJKLMNOPQRSTUVWXYZ'
phTerm=' {} '

ampDict={'&apos': '\'','&quot': '\"','&amp': 'm','&gt': '>','&lt': '<'}
        # m has some width as &, as we cannot insert &

cellJustify={
        'c':    'centre',
        'l':    'start',
        'r':    'end'
}

def calcCharWid():
  "calculate character widths from charCount dictionary"
  "scale is centimetre times 1000"
  pw=17000.0                    # width between test margins in cm * 1000
  pad=1.0*pw/charCount[0][0]    # width of '|' padding char
  i=32                          # start with ' '
  while i in charCount:
    w=(pw-pad*charCount[i][1])/(1.0*charCount[i][0])
        # subtract padding required to fill line and then divide by chars in line
    gv.charWid[i]=round(w,1)
    i=i+1
  gv.spaceW=gv.charWid[32]
  # calcCharWid

# the following produce format errors if orphaned

mathsOrphans=['-','+',
        'acute','bar','bold','breve','circle','check','dot','ddot','dddot','fact',
        'grave','hat','iint','iiint','int','ital','lim','lint','lline','llint',
        'lllint','neg','ortho','overline','overstrike','parallel','prod','prop',
        'setminus','slash','sqrt','sum','tilde','underline'
]

binOps=['=','<=','>=','<>','&lt;=','&gt;=','&lt;&gt;','and','div','divides',
        'equiv','intersection','nsubset','nsubseteq','nsupset','nsupseteq',
        'ndivides','notin','or','over','owns','sim','simeq','subset','subseteq',
        'supset','supseteq','times','union'
]

compl={ '(':    ')',
        '[':    ']',
        '|':    '|',
        'l':    'r'
}

nonEqnEsc={
        'dotslow':      '. . .'
}

LeftBR=['(','[','langle','lbrace','lceil','ldbracket','ldline','ldline','lline']

#---------------------------------------------------------------------------------
"the following tables and dictionaries are for the maths syntax checker "

leftBR={'{':            '"{"',
        '(':            '"("',
        '[':            '"["',
        '\{':           '"{"',
        'langle':       '{} lt {}',
        'lbrace':       '"{"',
        'lceil':        '"["',
        'ldbracket':    '"[["',
        'ldline':       '"||"',
        'lfloor':       '"["',
        'lline':        '"|"'
}
rightBR={'}':           '"}"',
        ')':            '")"',
        ']':            '"]"',
        '\}':           '"}"',
        'rangle':       '{} gt {}',
        'rbrace':       '"}"',
        'rceil':        '"]"',
        'rdbracket':    '"]]"',
        'rdline':       '"||"',
        'rfloor':       '"]"',
        'rline':        '"|"'
}
matchingBR={
        '{':            '}',
        '(':            ')',
        '[':            ']',
        '\{':           '\}',
        'langle':       'rangle',
        'lbrace':       'rbrace',
        'lceil':        'rceil',
        'ldbracket':    'rdbracket',
        'ldline':       'rdline',
        'lfloor':       'rfloor',
        'lline':        'rline'
}

unop=['sum','prod','lim','+','-','+-','-+','neg','cprod','int','iint','iiint',
        'lint','liint','liiint']
binop=[ '+','-','*','/','=','<','>','<>','<=','>=','&lt;','&gt;','&lt;=','&gt;=',
        'div','times','over','div','circ',
        'cdot','wideslash','widebslash','and','or','leslant','geslant',
        'approx','simeq','sim','equiv','prop','parallel','ortho','divides',
        'ndivides','toward','dlarrow','drarrow','dlrarrow','in','notin','owns',
        'intersection','union','setminus','slash','subset','subseteq','supset',
        'supseteq','nsubset','nsubseteq','nsupset','nsupseteq']
mustQuote=['binom','bold','Bold','font','from','ital','Ital','italic','Italic',
        'left','over','right','serif','size','to']
symbols=['infinity','partial','nabla','exists','forall','hbar','lambdabar','Re','Im',
        'wp','leftarrow','rightarrow','uparrow','downarrow','dotslow','dotsaxis',
        'dotsvert','dotsup','dotsdown','newline','alignl','alignc','alignr','`','~']
attributes=['acute','grave','check','breve','circle','dot','ddot','dddot','bar',
        'vec','tilde','hat','widevec','widetilde','widehat','overline','underline',
        'overstrike','phantom','bold','Bold','ital','Ital','Italic','italic',
        'overbrace','underbrace']
functions=['abs','fact','cos','cosh','cot','coth','exp','ln','log','sin','sinh',
        'sqrt','tan','tanh']
subsup=['_','^','lsup','csup','lsub','csub']

setCentral={'_': ' csub','^': ' csup'}

ops='+-*/^_=<>&'
Ops=['+','-','<','>','=','+-','-+','<>','<=','>=']

"""     this table relates special handling function names to object names
        it must be kept in synchronisation with objDict in TexConv.py
        the element function is that for items within a composite one
        the obj function is for handling a composite object bracketted by
        StartObject and EndObject
        the parameter 'major' controls the action of saveContext()
                if 0 the text obtained must be combined with that previously parsed
                if 1 not so
                if 2 finishPara() and handleBlankLine() are not executed at start of
                        object
"""

specFunc={
        #       name                    element function        obj function   major
        0:      ['parent',              'handleError'],
        1:      ['Abstract',            'handleElement',        'Abstract',     1],
        2:      ['Address',             'handleElement',        'handleObjBody',1],
        3:      ['Array',               'handleElement',        'Array',        2],
        4:      ['Author',              'handleElement',        'Author',       1],
        5:      ['Bibliography',        'handleElement',        'Bibliog',      1],
        6:      ['Bibliography label',  'handleNull',           'BibLabel',     0],
        7:      ['Lyxcode',             'handleElement',        'Lyxcode',      1],
        8:      ['Caption',             'handleCaption',        'Caption',      0],
        9:      ['Chapter heading',     'handleElement',        'Chapter',      1],
        10:     ['spare01',             'handleError'],
        11:     ['Comment',             'handleNull',           'Comment',      0],
        12:     ['Cross reference',     'handleElement'],
        13:     ['Date',                'handleElement',        'Date',         1],
        14:     ['Description',         'handleElement',        'Description',  1],
                                        # see note for Lyx list
        15:     ['Document structure',  'handleError'],
        16:     ['EndObject',           'handleEnd'],
        17:     ['ensuremath',          'handleEnsureMath',     'handleObjBody',3],
        18:     ['Enumerated list',     'handleElement',        'insetObject',  1],
        19:     ['Equation',            'handleElement',        'Equation',     1],
        20:     ['Eqn array',           'handleError',          'EqnArray',     1],
        21:     ['Eqn array numbered',  'handleElement',        'EqnArray',     1],# must be handleElement for equation label
        22:     ['Eqn inline',          'handleElement',        'EqInline',     3],
        23:     ['Equation label',      'handleEqnLabel'],
        24:     ['Escape char',         'handleEsc'],
        25:     ['Figure',              'handleElSize',         'Figure',       1],
        26:     ['Font size',           'handleFontSize'],
        27:     ['Font style',          'handleFontStyle'],
        28:     ['Footnote',            'handleElement',        'Footnote',     1],
        29:     ['Format',              'handleFormat'],
        30:     ['Fracdenom',           'handleElSize',         'FracDenom',    2],
        31:     ['Fraction',            'handleElSize',         'Fraction',     2],
        32:     ['Graphics',            'handleGraphics',       'Graphics',     1],
        33:     ['Itemised list',       'handleElement',        'insetObject',  1],
        34:     ['Label',               'handleLabel'],
        35:     ['List item',           'handleListItem',       'handleObjBody',0],
        36:     ['Lyx-style list',      'handleElement',        'LyxList',      1],
                                        # handleElement covers embedded paragraphs
                                        # handleListItem not used (c.f. use of '[')
        37:     ['Margin note',         'handleMarginNote'],
        38:     ['Multicolumn',         'handleElSize',         'MultiCol',     0],
        39:     ['Numbered paragraph',  'handleElement',        'handleObjBody',1],
        40:     ['Numbered part',       'handlePart',           'Part',         1],
        41:     ['Numbered section',    'handleElement',        'Section',      1],
        42:     ['Numbered subparagraph','handleElement',       'SubSubSection',1],# YES!
        43:     ['Numbered subsection', 'handleElement',        'SubSection',   1],
        44:     ['Numbered subsubsection','handleElement',      'SubSubSection',1],
        45:     ['Object parameters',   'handleParams'],
        46:     ['Page ref',            'handleElement',        'handleObjBody',3],
        47:     ['spare2',              'handleError'],
        48:     ['Parbox',              'handleNull',           'handleObjBody',1],
        49:     ['Quotation',           'handleQuotation',      'Quotation',    1],
        50:     ['Quote',               'handleQuote',          'Quote',        1],
        51:     ['Right address',       'handleElSize'  ,       'RightAddr',    1],
        52:     ['Sideways',            'handleElement',        'Sideways',     0],
        53:     ['StartObject',         'handleStart'],
        54:     ['Table',               'handleElement',        'Table',        1],
        55:     ['Table of contents',   'handleTOC'],
        56:     ['Tabular',             'handleElSize',         'Tabular',      1],
        57:     ['Text',                'handleText'],
        58:     ['Title',               'handleElement',        'Title',        1],
        59:     ['Unnumbered paragraph','handleElement',        'handleObjBody',1],
        60:     ['Unnumbered part',     'handleElement',        'Part',         1],
        61:     ['Unnumbered section',  'handleElement',        'Section',      1],
        62:     ['Unnumbered subparagraph','handleElement',     'SubSubSection',1],# YES!
        63:     ['Unnumbered subsection','handleElement',       'SubSection',   1],
        64:     ['Unnumbered subsubsection','handleElement',    'SubSubSection',1],
        65:     ['Unrecognised',        'handleUnrecog'],
        66:     ['URL',                 'handleURL',            'URL',          3],
        67:     ['Verse',               'handleVerse',          'Verse',        1],
        68:     ['vspace',              'handleVspace'] # must be last object
      }

def make_xRef():
  "make reversed cross ref from names in specFunc to its indices"
  for i,L in specFunc.iteritems():
    if L[2:]:
      gv.xRef[L[0]]=L[2],L[3]
  # make_xRef

def r(x):
  "round x to 3 decimal places and return as a string"
  return str(round(x,3))
  # r

def reportErr():
  "report error on terminal"
  ff=open(ErrPath)
  s=ff.read()
  ff.close()
  print s
  textGUI(s)
  # reportErr

class objectContext:
  "specifies machine state"
  "e.g. 'standard' 'table' 'eqn' 'quote' 'lyxlist' 'descr' or 'fig'"
  state='standard'
  prev=[]

  def set(self,objType):
    self.prev.append(self.state)
    self.state=objType
    # set

  def unset(self):
    self.state=self.prev.pop()
    # unset

  def getPrev(self):
    return self.prev[-1]
    # getPrev

  # objectContext class

ObjContext=objectContext()

def makeContentMain():
  "make main content.xml file"
  try:
    ipf=open(xmlDir+'contentMain.xml')
  except:
    makeErrorFile('Cannot open: contentMain.xml')
    return 0
  try:
    opf=open(docDir+'content.xml','wt')
  except:
    makeErrorFile('Cannot create: main content.xml')
    ipf.close()
    return 0
  s=ipf.readline()
  while s:
    if s[:15] == '<!-- PARAMETERS':
      if s[16:33] == 'style definitions':
        opf.write(gv.contentAutoStyles)
      elif s[16:23] == 'deffont':
        if ts.defFont != 'Times New Roman':
          opf.write(ts.defFontDecl[0])
          opf.write(ts.defFontDecl[2])
      else:
        i=0
        while gv.contentBody[i:]:
          opf.write(gv.contentBody[i])
          i=i+1
    elif s[:4] == '<!--':
      pass                      # do not copy comment lines
    else:
      opf.write(s[:-1])         # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()
  return 1
  # makeContentMain

def makeStyles():
  "make styles.xml file"
  try:
    ipf=open(xmlDir+'styles.xml')
  except:
    makeErrorFile('Cannot open: styles.xml')
    return 0
  try:
    opf=open(docDir+'styles.xml','wt')
  except:
    makeErrorFile('Cannot create: styles.xml')
    ipf.close()
    return 0
  s=ipf.readline()
  size=str(ts.defFontSize)

  while s:
    if s[:14] == '<!-- PARAMETER':
      if s[15:18] == 'tab':
        opf.write(str(ts.tabStop))
      elif s[15:24] == 'pagewidth':
        opf.write(ts.PageWidth)
      elif s[15:25] == 'pageheight':
        opf.write(ts.PageHeight)
      elif s[15:26] == 'orientation':
        opf.write(ts.orientation)
      elif s[15:24] == 'topmargin':
        opf.write(ts.TopMargin)
      elif s[15:25] == 'bottmargin':
        opf.write(ts.BottomMargin)
      elif s[15:25] == 'leftmargin':
        opf.write(ts.LeftMargin)
      elif s[15:26] == 'rightmargin':
        opf.write(ts.RightMargin)
      elif s[15:21] == 'mirror':
        opf.write(ts.mirror)
      elif s[15:22] == 'deffont':
        if ts.defFont != 'Times New Roman':
          opf.write(ts.defFontDecl[0])
          opf.write(ts.defFontDecl[1])
          opf.write(ts.defFontDecl[2])
      elif s[15:23] == 'fontname':
        opf.write(ts.defFont)
      else:
        opf.write(size)
    elif s[:4] == '<!--':
      pass                      # do not copy other comment lines
    else:
      opf.write(s[:-1])         # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()
  # makeStyles

def makeMeta():
  "make meta.xml file"
  try:
    ipf=open(xmlDir+'meta.xml')
  except:
    makeErrorFile('Cannot open: meta.xml')
    return 0
  try:
    opf=open(docDir+'meta.xml','wt')
  except:
    makeErrorFile('Cannot create: meta.xml')
    ipf.close()
    return 0
  s=ipf.readline()
  t=datetime.date.today()
  datenow=str(t.year)+'-'+str(t.month)+'-'+str(t.day)+'T00:00:00'
  while s:
    if s[:15] == '<!-- PARAMETER ':
      if s[15:20] == 'datec':
        opf.write(datenow)
      elif s[15:20] == 'daten':
        opf.write(datenow)
      elif s[15:21] == 'tables':
        opf.write(str(gv.tabN))
      elif s[15:21] == 'images':
        opf.write(str(gv.picN))
      elif s[15:22] == 'objects':
        opf.write(str(gv.objN))
    elif s[:4] == '<!--':
      pass                      # do not copy comment lines
    else:
      opf.write(s[:-1])         # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()
  # makeMeta

def makeManifest():
  "make manifest.xml file"
  subDir='META-INF'
  os.mkdir(docDir+subDir)
  try:
    ipf=open(xmlDir+'manifest.xml')
  except:
    makeErrorFile('Cannot open: manifest.xml')
    return 0
  try:
    opf=open(docDir+subDir+'/manifest.xml','wt')
  except:
    makeErrorFile('Cannot create: manifest.xml')
    ipf.close()
    return 0
  s=ipf.readline()
  while s:
    if s[:14] == '<!-- PARAMETER':
      if s[15:19] == 'pics':
        if gv.picManifest:
          opf.write(gv.picManifest)
      if s[15:19] == 'eqns':
        if gv.eqnManifest:
          opf.write(gv.eqnManifest)
    elif s[:4] == '<!--':
      pass                      # do not copy comment lines
    else:
      opf.write(s[:-1])         # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()
  # makeManifest

def copyMimetype():
  "make mimetype file"
  "see example in ../../Dictionaries/OfficeXML"
  try:
    ipf=open(xmlDir+'mimetype')
  except:
    makeErrorFile('Cannot open: mimetype')
    return 0
  try:
    opf=open(docDir+'mimetype','wt')
  except:
    makeErrorFile('Cannot create: mimetype')
    ipf.close()
    return 0
  s=ipf.read()
  opf.write(s)
  ipf.close()
  opf.close()
  # copyMimetype

def makeSettingsMain():
  "make main settings.xml file"
  "do this when the above parameters have been set"
  gv.settingsMain['PrinterSetup']=printerSetup
  try:
    ipf=open(xmlDir+'settingsMain.xml')
  except:
    makeErrorFile('Cannot open: settingsMain.xml')
    return 0
  try:
    opf=open(docDir+'settings.xml','wt')
  except:
    makeErrorFile('Cannot create: main settings.xml')
    ipf.close()
    return 0
  s=ipf.readline()
  while s:
    if s[:14] == '<!-- PARAMETER':
      s=s[15:-5]                # parse parameter name excluding trailing ' -->\n'
      opf.write(gv.settingsMain[s]) # write actual param from dictionary
    elif s[:4] == '<!--':
      pass                      # do not copy comment lines
    else:
      opf.write(s[:-1])         # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()
  # makeSettingsMain

def makeSettingsEqn(diry):
  "make settings.xml file for an equation"
  "do this when the above parameters have been set"
  try:
    ipf=open(xmlDir+'settingsEqn.xml')
  except:
    makeErrorFile('Cannot open: settingsEqn.xml')
    return 0
  try:
    opf=open(docDir+diry+'/settings.xml','wt')
  except:
    makeErrorFile('Cannot create: settings.xml in '+docDir+diry)
    return 0
  s=ipf.readline()
  while s:
    if s[:14] == '<!-- PARAMETER':
      s=s[15:-5]                # parse parameter name excluding trailing ' -->\n'
      opf.write(gv.settingsEqn[s])      # write actual param from dictionary
    elif s[:4] == '<!--':
      pass                      # do not copy comment lines
    else:
      opf.write(s[:-1])         # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()
  return 1
  # makeSettingsEqn

def makeContentEqn(diry):
  "makes contents.xml file for an equation and writes manifest entry"
  "do this when the equation has been parsed into StarMath format and"
  "the sizes have been calculated"

  # make equation content file -----------------------------------------------

  try:
    ipf=open(xmlDir+'contentEqn.xml')
  except:
    makeErrorFile('Cannot open: contentEqn.xml')
    return 0
  try:
    opf=open(docDir+diry+'/content.xml','wt')
  except:
    makeErrorFile('Cannot create: content.xml in '+docDir+diry)
    return 0
  s=ipf.readline()
  while s:
    if s[:14] == '<!-- PARAMETER':
      opf.write(th.text)
    elif s[:4] == '<!--':
      pass                      # do not copy comment lines
    else:
      opf.write(s[:-1])                 # omit trailing '\n'
    s=ipf.readline()
  ipf.close()
  opf.close()

  # do manifest ------------------------------------------------------------------

  s='<manifest:file-entry manifest:media-type="'
  s1='" manifest:full-path="'+diry+xmlSep
  gv.eqnManifest=gv.eqnManifest+s+'text'+xmlSep+'xml'+s1+'content.xml"/>'
  gv.eqnManifest=gv.eqnManifest+s+'text'+xmlSep+'xml'+s1+'settings.xml"/>'
  gv.eqnManifest=gv.eqnManifest+s+'application'+xmlSep+'vnd.sun.xml.math'+s1+'"/>'
  return 1
  # makeContentEqn

PDict={ # fonts available: 0x24000000, 0x280000000, 0x2c000000, 0x30000000,
        # 0x34000000, 0x38000000 and 0x3c000000
        'keep':         0x40000000,     # keep with next para
        'fontMask':     0x3c000000,     # to reset font type
        'grph':         0x20000000,     # centred graphics
        'eqnl':         0x1c000000,     # labelled equations
        'eqnc':         0x18000000,     # centred equations
        'sserif':       0x14000000,     # sans serif
        'tilde':        0x10000000,     # tilde, umlaut, macron and Hungarian umlau
        'acute':        0xc000000,      # acute
        'symbol':       0x8000000,      # symbol
        'ttype':        0x4000000,      # Courier
        'justifyMask':  0x3000000,      # for resetting justification
        'end':          0x3000000,
        'start':        0x2000000,
        'centre':       0x1000000,
        'justify':      0,
        'italic':       0x800000,               # set if italic
        'bold':         0x400000,               # set if bold
        'single':       0x200000,               # set for single underline
        'posMask':      0x180000,               # for resetting position (sub or super)
        'sub':          0x180000,               # sub and super cannot both be set
        'super':         0x80000,
        'indentMask':    0x7e000,               # indentation mask
        'marginMask':     0x1f80,               # margin mask
        'normal':       0,
        'none':         0,
        0x40000000:     'keep',
        0x20000000:     'grph',
        0x1c000000:     'eqnl',
        0x18000000:     'eqnc',
        0x14000000:     'style:font-name="Sans Serif" ',
        0x10000000:     'style:parent-style-name="Lucidabright1" ',
        0xc000000:      'style:parent-style-name="Lucidabright1" ',
        0x8000000:      'style:parent-style-name="Lucidabright1" ',
        0x4000000:      'style:font-name="Courier" ',
        0x3000000:      'end',
        0x2000000:      'start',
        0x1000000:      'center',
        0x800000:       'italic',
        0x400000:       'bold',
        0x200000:       'single',
        0x180000:       'sub',
        0x80000:        'super',
        0:              'normal',
}

class textStyle:
  "class to handle text styles"
  "the variable currentStyle is a binary combination of the style attribute bits"
  "defined in PDict{}; it  controls the construction of font style declarations in"
  "the main content.xml file;"
  "the attribute variables in previous versions have been removed as conflict could"
  "easily arise between them and currentStyle; the latter is now the sole arbiter"
  "of all style handling;"
  "originally it was used as the P or T style name number for OpenOffice, which"
  "worked but apparently caused some problems for OpenOffice as the document was"
  "reformatted shortly after loading it;  therefore the policy was changed to"
  "use ascending relatively small style name numbers starting with P1 and T1;"
  "the dictionaries styleDict and styleRDict are therefore constructed to relate"
  "the two, as for global consistency we retain currentStyle as the sole arbiter of"
  "style handling i.e. it and only it is referred to when changing the style or"
  "looking up style attributes"
  """ externally called methods are:
        calcFactor()
        getFont()
        getIndent()
        getMargin()
        getSize()
        incrMargin()
        initStyle()             # call at start of program to set default
        reduceMargin()
        removeMargin()
        restoreStyle()
        saveStyle()
        setDefFont()
        setFont()               # set font type
        setIndent()
        setMargin()
        setJustify(justify)
        setKeep(keep)           # set/clear keep with next paragraph
        setSize(size)
        setStyle(style)
        setUnderline(underline)
        setWeight(weight)
      useful variables are:
        currentStyle
        defFontSize
        size
  """
  PageWidth='21'                # cm for A4
  PageHeight='29.7'             # cm for A4
  orientation='portrait'
  TopMargin='2'
  BottomMargin='2'
  LeftMargin='2'
  RightMargin='1'
  mirror=' '                    # set if for two-sided pages
  lineSpacing=''
  defFont='Times New Roman'     # default font
  defFontDecl=[]                # declaration of default font if not Times New Roman

  defFontSize=10
  fontScale=0.0001              # = size*subSup/12000
  default=''
  margins=[0.25*x for x in range(0,64)]
  indent=0.1                    # indent increment = 1 mm
  tabStop=2.0                   # tab stop size
  subSup=1                      # set 0.58 if in subscript or superscriptmode
  isSmallCap=0                  # 1 if noun style
  currentStyle=10               # bit-structured number defining current 'P' style
                                # initialised to all defaults, size 12pt
  newScale={'^': 0.58,'_': 0.58,'bold': 1.07}   # this is for equation scaling
  styleDict={'P': {},'T': {}}   # conversion between currentStyle and style name
  styleRDict={}                 # conversion between style name and currentStyle
  styleNum={'P': 1,'T':1}       # latest formal style numbers
  styleNames=[]
  TstyleNames=[]
  savedStyle=[]

  def initStyle(self):
    "called at start of program"
    self.currentStyle=self.defFontSize
    self.default=self.currentStyle
    self.setPStyle()
    self.fontScale=self.getSize()*self.calcFactor()/12000.0
    # initStyle

  def setDefFont(self):
    "set default style"
    self.currentStyle=self.default
    self.fontScale=0.0001
    self.subSup=1
    self.isSmallCap=0
    # setDefFont

  def makeStyleName(self,Type):
    "construct paragraph style name"
    "of form e.g. Pbbbbis if Type = 'P', where bbbb is binary combination,"
    "i is margin (4 bits) and s is pt size (7 bits)"
    if self.currentStyle in self.styleDict[Type]:
      return self.styleDict[Type][self.currentStyle]
    n=self.styleNum[Type]
    sn=Type+str(n)
    self.styleNum[Type]=n+1
    self.styleDict[Type][self.currentStyle]=sn
    self.styleRDict[sn]=self.currentStyle
    return sn
    # makeStyleName

  def saveStyle(self):
    "save current style parameters"
    self.savedStyle.append(self.currentStyle)
    self.savedStyle.append(self.subSup)
    # saveStyle

  def calcFactor(self):
    "return size scaling factor due to special font attributes"
    f=1
    if operator.and_(self.currentStyle,PDict['sserif']) == PDict['sserif']:
      f=1.1             # f*0.787 or 1.271
    if operator.and_(self.currentStyle,PDict['bold']):
      f=f*1.1
    return f
    # calcFactor

  def restoreStyle(self):
    "restore current style parameters"
    self.subSup=self.savedStyle.pop()
    self.currentStyle=self.savedStyle.pop()
    z=self.getSize()*self.calcFactor()
    self.fontScale=z*self.subSup/12000.0
                # for conversion of lengths to cm
    # restoreStyle

  def getAttrib(self,s):
    "s is an ascii atribute mask name; the bit pattern for it is obtained from"
    "PDict and masked with the current style to give the bit pattern of the current"
    "style atribute; this is converted back to the ascii name of the current"
    "attribute which is returned"
    return PDict[operator.and_(self.currentStyle,PDict[s])]
    # getAttrib

  def setAttrib(self,s):
    self.currentStyle=operator.or_(self.currentStyle,PDict[s])
    # setAttrib

  def resetAttrib(self,s):
    "e.g. reset font bits to all zeros if s='fontMask'"
    self.currentStyle=operator.and_(self.currentStyle,operator.inv(PDict[s]))
    # resetAttrib

  def styleDetails(self):
    x=self.getSize()
    fs='fo:font-size="'+str(x)+'pt" '
    z=self.getAttrib('italic')
    fst='fo:font-style="'+z+'" '
    x=self.getAttrib('single')
    if x == 'single':
      fu='style:text-underline="single" '
      fu=fu+'style:text-underline-color="font-color" '
    else:
      fu='style:text-underline="none" '
    z=self.getAttrib('bold')
    fw='fo:font-weight="'+z+'" '
    ass=fs[:12]+'-asian'+fs[12:]
    ast=fst[:13]+'-asian'+fst[13:]
    aw=fw[:14]+'-asian'+fw[14:]
    cs=fs[:12]+'-complex'+fs[12:]
    ct=fst[:13]+'-complex'+fst[13:]
    cw=fw[:14]+'-complex'+fw[14:]
    z=self.getAttrib('justifyMask')
    if z == 'normal':
      z='justify'
    fj='fo:text-align="'+z+'" style:justify-single-word="false"'
    if self.getAttrib('keep') == 'keep':
      fj=fj+' fo:keep-with-next="true"'
    fj=fj+'/>'
    return fs+fst+fu+fw+ass+ast+aw+cs+ct+cw+fj
    # styleDetails

  def setPStyle(self):
    "construct style definition"
    if self.currentStyle in self.styleNames:
      return
    fn=self.makeStyleName('P')
    s='<style:style style:name="'+fn+'" '
    s=s+'style:family="paragraph" style:parent-style-name="Standard">'
    f='<style:properties '+self.lineSpacing
    z=self.getAttrib('fontMask')
    if z == 'tilde' or z == 'acute':
      f=f+'style:font-name-asian="1980 Portable" style:font-name-complex="1980 Portable" '
    z=self.getMargin()
    if z:
      f=f+'fo:margin-left="'+r(self.margins[z])+'cm" fo:margin-right="0cm" '
    z=self.getIndent()
    if z:
      f=f+'fo:text-indent="'+r(z*self.indent)+'cm" '
    z=self.getAttrib('posMask')
    if z != 'normal':
      f=f+'style:text-position="'+z+' 58%" '
      self.subSup=0.58
    else:
      self.subSup=1
    z=self.getAttrib('fontMask')
    if z == 'normal':
      f=f+'style:font-name="'+ts.defFont+'" '
    else:
      f=f+z
    gv.contentAutoStyles=gv.contentAutoStyles+s+f+self.styleDetails()
    gv.contentAutoStyles=gv.contentAutoStyles+'</style:style>'
    z=self.getSize()*self.calcFactor()
    self.fontScale=z*self.subSup/12000.0        # for conversion of lengths to cm
    self.styleNames.append(self.currentStyle)
    # setPStyle

  def setTStyle(self):
    "construct T style if needed"
    "assumes P style is current and makes T style from it"
    if self.currentStyle in self.TstyleNames:
      return
    fn=self.makeStyleName('T')
    if not self.currentStyle in self.styleNames:
      self.setPStyle()
    s='<style:style style:name="'+fn+'" style:family="text"><style:properties '
    s=s+self.lineSpacing
    z=self.getMargin()
    if z:
      s=s+'fo:margin-left="'+r(self.margins[z])+'cm" fo:margin-right="0cm" '
    z=self.getIndent()
    if z:
      s=s+'fo:text-indent="'+r(z*self.indent)+'cm" '
    z=self.getAttrib('posMask')
    if z != 'normal':
      s=s+'style:text-position="'+z+' 58%" '
      self.subSup=0.58
    else:
      self.subSup=1
    z=self.getAttrib('fontMask')
    if z == 'normal':
      s=s+'style:font-name="Times New Roman1" '
    else:
      s=s+z
    gv.contentAutoStyles=gv.contentAutoStyles+s+self.styleDetails()
    gv.contentAutoStyles=gv.contentAutoStyles+'</style:style>'
    z=self.getSize()*self.calcFactor()
    self.fontScale=z*self.subSup/12000.0  # for conversion of lengths to cm
    self.TstyleNames.append(self.currentStyle)
    # setTStyle

  def setSize(self,deltasize):
    "size usually 12 or 16"
    size=self.defFontSize+deltasize
    z=self.getSize()
    if size != z:
      self.currentStyle=self.currentStyle-z+size
    # setSize

  def getSize(self):
    return operator.and_(self.currentStyle,0x7f)
    # getSize

  def getMargin(self):
    return operator.and_(self.currentStyle,PDict['marginMask'])/0x80
    # getMargin

  def setMargin(self,n):
    self.resetAttrib('marginMask')
    if n>63:
      n=63
    self.currentStyle=operator.or_(self.currentStyle,n*0x80)
    # setMargin

  def setFont(self,s):
    self.resetAttrib('fontMask')
    self.setAttrib(s)
    # setFont

  def getFont(self):
    return self.getAttrib('fontMask')
    # getFont

  def setStyle(self,style):
    "style = 'normal' or 'italic', but others handled here"
    z=self.getAttrib('italic')
    if style != z:
      if style == 'ttype':
        self.setFont('ttype')
        self.setSize(-1)
      elif style == 'Lyx':              # lyxcode
        self.setFont('ttype')
        self.setSize(-1)
      elif style == 'normal':
        self.resetAttrib('fontMask')    # reset implies default font
      elif style == 'smcap':
        size=self.defFontSize-2         # set size for small caps
        self.currentStyle=self.currentStyle-self.getSize()+size
        self.isSmallCap=1
      elif style[0] == '_':
        self.resetAttrib('posMask')
        self.setAttrib('sub')
      elif style[0] == '^':
        self.resetAttrib('posMask')
        self.setAttrib('super')
      elif style == 'sserif':
        self.resetAttrib('italic')
        self.setAttrib(style)
        ts.fontScale=ts.fontScale*1.2
      else:
        self.resetAttrib('italic')
        self.setAttrib(style)
    # setStyle

  def setWeight(self,weight):
    "weight= 'normal' or 'bold'"
    z=self.getAttrib('bold')
    if weight != z:
      self.resetAttrib('bold')
      self.setAttrib(weight)
      if weight == 'bold':
        ts.fontScale=ts.fontScale*1.1
    # setWeight

  def setUnderline(self,underline):
    "underline= 'none' or 'single'"
    z=self.getAttrib('single')
    if z == 'normal':
      z='none'
    if underline != z:
      self.resetAttrib('single')
      self.setAttrib(underline)
    # setUnderline

  def setKeep(self,keep):
    "set/clear keep with next paragraph attribute"
    "keep = 'keep' or 'normal'"
    z=self.getAttrib('keep')
    if keep != z:
      self.resetAttrib('keep')
      self.setAttrib(keep)
    # setKeep

  def setPosition(self,position):
    "position = 'normal' or 'sub' or 'super'"
    z=self.getAttrib('posMask')
    if position != z:
      self.resetAttrib('posMask')
      self.setAttrib(position)
    # setPosition

  def setJustify(self,justify):
    "justify = 'justify' or 'end' or 'start' or 'centre'"
    z=self.getAttrib('justifyMask')
    if justify != z:
      self.resetAttrib('justifyMask')
      if justify == 'justify':
        justify='normal'
      self.setAttrib(justify)
    return z
    # setJustify

  styleSpec={# translates accent into required style
        'macron':       'tilde',
        'tilde':        'tilde',
        'hung':         'tilde',
        'cedil':        'symbol',
        'circ':         'symbol',
        'acute':        'acute',
        'grave':        'acute'
  }

  def setAccent(self,style):
    "accented chars from ISO 8859 2 - 4: tilde, macron, Hungarian umlaut"
    z=self.getAttrib('fontMask')
    if style != z:
      self.setFont(self.styleSpec[style])
    doAccent(style)
    # setAccent

  def incrMargin(self,n):
    "increase margin of current style"
    n=n+self.getMargin()
    self.setMargin(n)
    # incrMargin

  def reduceMargin(self,n):
    "reduce margin of current style"
    n=self.getMargin()-n
    if n<0:
      n=0
    self.setMargin(n)
    # reduceMargin

  def removeMargin(self):
    "remove margin"
    self.setMargin(0)
    # removeMargin

  def removeIndentation(self):
    "remove indentation"
    self.setIndent(0)
    # removeIndentation

  def setIndent(self,n):
    "set indent to n units"
    self.resetAttrib('indentMask')
    i=abs(n)
    if i>31:
      i=31
    if n<0:                             # if is negative
      i=i+0x20                          # then set sign bit
    self.currentStyle=operator.or_(self.currentStyle,i*0x2000)
    # setIndent

  def getIndent(self):
    z=operator.and_(self.currentStyle,PDict['indentMask'])/0x2000
    if z>0x1f:          # if sign bit set
      z = 0x20-z        # make negative
    return z
    # getIndent

  # textStyle class

ts=textStyle()

def getCharWid(s):
  "get width of char in cm x 1000 for 12pt font size"
  c=ord(s[0])
  c1=32
  if s[1:]:
    c1=ord(s[1])
  i=1
  if c in nonAscii and len(s)>1 and c1 != 32:   # if not ascii
    if c<0xe0:
      if c1 in nonAscii[c]:
        w=nonAscii[c][c1]               # then get its width (2 hex bytes)
      else:
        w=200
        #print hex(c),hex(c1)
      i=2
    else:
      w=nonAscii[c][c1][ord(s[i+1])]    # or 3 hex bytes
      i=3
  elif ts.getFont() == 'ttype':
    w=191
  elif c in gv.charWid:                 # if ascii
    w=gv.charWid[c]*gv.textF            #  get ascii width
  else:
    w=200                               # else assume 200
  return w,i
  # getCharWid

def textLength(s):
  "calculate text length of s in cm*1000 for 12pt font"
  i=s.find('&')
  while i != -1:
    j=s[i:].find(';')+i
    s=s[:i]+ampDict[s[i:j]]+s[j+1:]     # replace e.g. &lt; with <
    i=s.find('&')
  i=0
  w=0
  while s[i:]:
    L=getCharWid(s[i:])
    w=w+L[0]
    i=i+L[1]
  return w
  # textLength

class TextHandling:
  "handle entering text into gv.contentBody"
  """ Functions are:
        enterText       enter text in self.text according to current context
        finishPara      finish current para including embedded spans
        finishSpan      finish span if any, para not finished
        finishStyle     finish current style, span only if exists else para
        enterVspace     enter vertical space
        extractSpan     extract span from currentPara

  Indenting standard environments paragraphs is tricky;
        enterText() checks for this, and increments a counter which must be 2
        before indenting begins;
        that counter (paraCount) is reset by:
                enterText() if not in standard environment
                finishPara() likewise
                table, figure and equation objects
  """

  text=''               # current object text
  textState=''          # is '' or textPara ( = '<text:p text:style-name="')
                        # or textSpan ( = '<text:span text:style-name="')
  currentPara=''        # current paragraph being accumulated
                        # allows change on the fly e.g. if font style changes
  spanDepth=0           # now only goes to 1 max
  hasSpans=0            # true if current para has spane items
  lineWidth=78          # this is chars
  paraCount=0           # paragraph count in standard environment

  def enterText(self,isNew):
    "enter text in contentBody"
    "if paragraph not started set para context and write para bracket & text,"
    "   '</text:p>' is then required later"
    "if a paragraph has been started then write a span bracket"
    "   a matching '</text:span>' is then required later"
    if not self.text:
      return
    s=''
    if ObjContext.state == 'standard':
      th.paraCount=th.paraCount+1
      if th.paraCount>2:                # if is not first srandard para
        ts.setIndent(6)                 # then indent it
    else:
      th.paraCount=0
    gv.BlankInserted=0                  # reset blank line flag
    if not self.textState:
      self.textState=textPara
      ts.setPStyle()                    # construct P style if not available
      s=textPara+ts.makeStyleName('P')+'">'
      self.spanDepth=0
      self.hasSpans=0
    elif self.textState == textPara:
      if isNew or self.hasSpans:
        self.textState=textSpan
        ts.setTStyle()                  # construct T style if not available
        s=textSpan+ts.makeStyleName('T')+'">'
        self.spanDepth=1
        self.hasSpans=1
    else:       # is already in a span
      if isNew:
        s='</text:span>'
        ts.setTStyle()                  # construct new T style if not available
        s=s+textSpan+ts.makeStyleName('T')+'">'
        self.spanDepth=1
    if ts.isSmallCap:
      self.text=self.text.upper()
    self.currentPara=self.currentPara+s+self.text
    self.text=''
    # enterText

  def contSpan(self):
    "force a new span without starting a paragraph"
    "use is e.g. for footnotes where </text:p> has been removed"
    self.text=''
    self.textState=textPara
    # contSpan

  def calcSize(self,s):
    "calculate text size of s in cm, allowing for font size"
    cs=ts.currentStyle
    scale=ts.fontScale
    s=s.replace('\\hfill ','')
    w=0
    while s:
      i=s.find('<')
      if i == -1:
        w=w+textLength(s)*scale
        ts.currentStyle=cs
        return w
      if i:
        w=w+textLength(s[:i])*scale
        s=s[i:]
      i=s.find('>')
      j=s[:i].find('text:style-name')
      if j > -1:
        j=j+17
        k=s[j:].find('"')+j
        ts.currentStyle=ts.styleRDict[s[j:k]]
        scale=ts.getSize()*ts.calcFactor()/12000.0
      s=s[i+1:]
    ts.currentStyle=cs
    return w
    # calcSize

  def doHfill(self,width,text):
    "space pad separate text items between margins"
    "allow for margin, indent, spaces and inline equations as well as text"
    if width > gv.labelW:
      x=ts.margins[ts.getMargin()]+ts.getIndent()*ts.indent
    else:
      x=0
    ss=gv.spaceW*ts.fontScale   # space size in cm
    s=text
    i=s.find('text:c="')
    while i != -1:              # check for space fillers already inserted
      s=s[i:]
      L=parseInt(s)
      x=x+L[0]*ss               # initialise x to any spaces from previous hfills
      s=L[1]                    # e.g. in a List label
      i=s.find('text:c="')
    s=text
    i=s.find('draw:style-name="fr1"')
    while i != -1:                      # check for equations
      i=s[i:].find('svg:width="')+i
      s=s[i:]
      L=parseReal(s)
      x=x+L[0]                          # add equation width to x
      i=s.find('draw:style-name="fr1"')
    x=x+self.calcSize(text)
    s=text
    n=s.count('\\hfill ')               # n = number of fills required
    f=1.0*(width-x)/n                   # f = fill length between text fragments
    i=int(f/ss+0.5)                     # i = number of filler spaces required
    if i<0:
      i=1
    filler=gv.hardspaces[:2*i]
    s=s.replace('\\hfill ',filler)      # replace every '\\hfill' with filler
    gv.isHfill=0                        # reset flag
    return s                            # and return result
    # doHfill

  def finishPara(self):
    "finish writing paragraph to contentBody"
    if not self.textState:      # if no text entered yet
      if not self.text:         # then if no text to enter
        if not self.currentPara:
          return                        # then abort
      else:
        self.enterText(0)               # else enter text
    if ts.isSmallCap:
      self.text=self.text.upper()
    s=self.text
    if self.spanDepth:          # handle outstanding span bracket
      s=s+'</text:span>'
      self.spanDepth=0
    elif self.hasSpans and s:
      sn=ts.makeStyleName('T')
      ts.setTStyle()
      s='<text:span text:style-name="'+sn+'">'+s+'</text:span>'
    self.currentPara=self.currentPara+s+'</text:p>'
    if gv.isHfill:                      # if hfills exist
      w=gv.pageWidth
      self.currentPara=self.doHfill(w,self.currentPara) # replace the hfills
    gv.contentBody.append(self.currentPara)
    self.currentPara=''
    self.textState=''
    self.text=''
    self.spanDepth=0
    self.hasSpans=0
    gv.BlankInserted=0                  # reset blank line flag
    if ObjContext.state == 'standard':  # if is standard para
      ts.setIndent(0)                   # then remove indentation
    else:
      th.paraCount=0
    # finishPara

  def addText(self):
    "close paragraph without writing </text:p>"
    if ts.isSmallCap:
      self.text=self.text.upper()
    self.currentPara=self.currentPara+th.text
    self.text=''
    self.spanDepth=0
    # addText

  def finishSpan(self):
    "finish outstanding span if there is one"
    "create new para if none started, and enter outstanding text"
    t=''
    if ts.isSmallCap:
      self.text=self.text.upper()
    if not self.textState and not self.currentPara:
      t='<text:p text:style-name="'+ts.makeStyleName('P')+'">'
      self.textState=textPara
    if ts.isSmallCap:
      self.text=self.text.upper()
    if self.textState != textSpan:
      if self.text:
        ts.setTStyle()
        t=t+'<text:span text:style-name="'+ts.makeStyleName('T')+'">'+self.text+'</text:span>'
    else:
      t=t+self.text+'</text:span>'
    self.currentPara=self.currentPara+t
    self.spanDepth=0
    self.textState=textPara
    self.text=''
    # finishSpan

  def finishStyle(self):
    "finish current style"
    self.finishSpan()
    ts.restoreStyle()
    self.spanDepth=0
    # finishStyle

  def convSpace(self,vs):
    "return size in cm of vspace = e.g. '0.3mm'"
    x=1.0*eval(vs[0:-2])        #ensure x is real
    if 'mm' in vs:
      x=x/10
    elif 'cc' in vs:    #assume means ciceros; 1 cc = 12.839 pt
      x=x*0.452641512   #12.839*0.01388*2.54
    elif 'pt' in vs:
      x=x*0.0352552     #0.01388*2.54
    elif 'in' in vs:
      x=x*2.54
    #default assume cm
    return x
    #convSpace

  def enterVspace(self,vspace):
    "enter vertical space by creating a style for it with 6pt size as minimum"
    "possible vspace (= 2.115312 mm) and bottom margin as excess above that"
    "and then write a paragraph in that style consisting of 2 spaces"
    "N.B. 1 pt = 0.01388 inches"
    x=self.convSpace(vspace)-0.2115312    #- 6pt in cm
    if x<0:
      x=0
    vs=r(x)+'cm'    #- 6pt in cm
    style='P'+vspace
    if not style in ts.styleNames:
      ts.styleNames.append(style)
      s='<style:style style:name="'+style+'" '
      s=s+'style:family="paragraph" style:parent-style-name="Standard">'
      s=s+'<style:properties fo:margin-left="0cm" fo:margin-right="0cm" '
      s=s+'fo:margin-top="0cm" fo:margin-bottom="'+vs+'" '
      s=s+'style:font-name="Times New Roman" fo:font-size="6pt" '
      s=s+'fo:font-style="normal" style:text-underline="none" fo:font-weight="normal" '
      s=s+'fo:font-size-asian="6pt" fo:font-style-asian="normal" '
      s=s+'fo:font-weight-asian="normal" fo:font-size-complex="6pt" '
      s=s+'fo:font-style-complex="normal" fo:font-weight-complex="normal" '
      s=s+'fo:text-align="justify" style:justify-single-word="false"/>'
      s=s+'</style:style>'
      gv.contentAutoStyles=gv.contentAutoStyles+s
    s='<text:p text:style-name="'+style+'"><text:s text:c="2"/>'
    # this is how to write two spaces!
    self.currentPara=self.currentPara+s
    self.finishPara()
    # enterVspace

  def extractSpan(self):
    "extract span from currentPara"
    self.finishPara()
    s=gv.contentBody.pop()
    i=s.find('<text:p ')
    if i == -1:
      return s
    i=s[i:].find('>')+i
    j=s.rfind('</text:p>')
    if j == -1:
      j=len(s)
    return s[i+1:j]
    # extractSpan
  # class TextHandling

th=TextHandling()

def removeErrFile():
  try:
    os.remove(ErrPath)                  # delete error file
  except:
    pass
  # removeErrFile

def makeErrorFile(msg):
  "make error file for client"
  removeErrFile()
  ff=open(ErrPath,'wt')
  ff.write(msg+'\n')
  ff.close()
  gv.syntaxErr=0
  # makeErrorFile

def rm_dirs(path):
  "remove non-empty directories as despite documentation removedirs() fails"
  fl=os.listdir(path)
  os.chdir(path)
  for f in fl:
    if os.path.islink(f):
      continue
    if os.path.isdir(f):
      rm_dirs(f)
    else:
      os.remove(f)
  os.chdir('..')
  try:
    os.rmdir(path)
  except:
    pass
  #rm_dirs

def makezip(root,files):
  "recursive procedure to add files in sub-dirs to a zip file"
  for f in files:
    r1=os.path.join(root,f)
    try:                    #this is a cheap way of testing if f is a subdir
      files1=os.listdir(f)
      os.chdir(f)               #test is is sub-directory name
      os.chdir('..')
      makezip(r1,files1)
    except:
      gv.zf.write(r1)              #add to zip file if not
  #makezip

def zipit():
  "zip utility for Windows environment"
  "assumes we are in the officeXML directory"
  if dirySep == '/':
    s='.sxw'
  else:
    s='.odt'
  gv.zf=zipfile.ZipFile(gv.targetDir+gv.docname+s,'w',8)   #create zip file
  # N.B. ZIP_DEFLATED is not recognised!!! (see documentation)
  # a trial loop found that 8 is the required value to deflate the file
  os.chdir(docDir)
  fl=os.listdir(docDir)
  makezip('',fl)
  gv.zf.close()
  #zipit

class globalVariables:
  "structure to hold global variables"
  ipf=0
  opf=0
  debug=0
  doIndent=0
  lastObject=0
  pageWidth=18          # pagewidth between margins cm
  spaceW=0              # space width in cm*1000 for 12pt
  textF=1               # factor set for text in equations
  hardspaces=''
  docname=''            # document name without path or sufffix
  targetDir=''          # where final file will be created
  fsd=None              # for diry GUI
  root=None
  zf=None

  lastParsed=()         # for maths syntax checker
  wasSS=False           # for maths syntax checker

  xRef={}
  textStack=[]

  # the objects in HoldBlank must not cause BlankInserted to be reset
  # i.e. objects in holdBlank must not prevent suppression of consecutive blank lines
  HoldBlank=['StartObject','EndObject','Font size','Font style','Format','Label']
  holdBlank=[]          # indices of above objects stored here

  objDict={}            # will hold unpickled object dictionary
  escapeDict={}         # escape char dictionary
  fontStyleDict={}      # for font style dictionary
  MathsStyleDict={}     # for maths font style dictionary
  fontSizeDict={}       # for font size dictionary
  formatDict={}         # for general format dictionary

  syntaxErr=0           # syntax error flag

  # items for constructing xml files
  eqnManifest=''        # manifest entries for equations
  picManifest=''        # manifest entries for pictures
  contentBody=[]        # for body of main contents.xml file
  contentAutoStyles=''  # for accumulating <office:automatic-styles> entries

  #variables for current equation ----------------------------------------------
  objDir='ObjBFFFDED'   # object dictionary base name
  eqnOffset=0           # cm x 1000
  eqnWidth=0            # cm x 1000
  eqnHeight=0           # cm x 1000
  eqnMargin=100         # cm x 1000
  eqnHtmargin=300       # cm x 1000
  eqnZI=0               # a sequence number for z-index
  isLabelled=0          # true if labelled
  EqnLab=''             # next equation label
  labelWidth=0          # next equation width in cm
  labFntSize=0          # font size of label to calculate space padding
  eqnStyle=''           # typically 'fr1' or 'fr2'
  settingsEqn={
        'ViewAreaWidth':        '',     # = str(eqnWidth)
        'ViewAreaHeight':       '',     # = str(eqnHeight)
}
  # special object variables ----------------------------------------------------

  lineLen=0             # length of a line e.g. for right address justification
  bibLab=0              # bibliography label number
  ftn=0                 # footnote number
  isHfill=0             # true if hfill found in line
  raStart=0             # right address start
  charWid={}            # character widths for 12pt Times New Roman font calculated
                        # from charCount dictionary
  preJust=''            # stores justify name prior to setting global centering
  arrCols=0             # number of columns of an array
  arrCol=0              # current array column number
  LLbrPend=0            # true if Lyx list label is unfinished
  labelWid=[]           # list label width stack
  labelN=4              # Lyx list label width in margin units
  labelW=2.0            # default label width
  lyxIndent=-20         # default indent for a Lyx List (-20 mm)



  # picture variables ------------------------------------------------------------
  picDirMade=0          # true if Picture subdirectory made
  pWid=0                # picture width
  pHt=0                 # picture height
  frameN=0              # frame number
  graphicN=0            # graphic number
  picName=''            # picture name
  picNum=0              # floated picture number
  capLen=0              # length of caption

  # table variables ---------------------------------------------------------------
  tableNo=0             # table number
  tableName=''          # table name
  tableWid=17           # table width
  maxColWid=0           # for sideways case to calc height of table for a float
  tableStart=0          # index into gv.contentBody where table entries atart
  tableText=''          #
  TableRow=''           # current row
  rowLines=0            # count of top lines of next row
  tableRows=0           # number of rows in table
  tableCols=0           # number of columns in table
  tableRow=0            # current row number, starting with 1
  cellStart=0           # is length of gv.contentBody before cell content is
                        # assembled; this allows normal action of format items to
                        # accumulate entries in gv.contentBody which can then be
                        # recovered and re-entered in proper context
  Cnames=[]             # for cell names such as 'A1'
  cellRow=[]            # cell default styles made from object parameters
  widthOveride=[]       # contains table column width overides
  hlineOveride=[]       # contains table horizontal line overides from \cline
  cellStyles={}         # dictionary of constructed cell styles e.g. 1.LDTS
  colWids={}            # column widths dictionary
  mergecols=0           # for multicolumn
  mergeformat=''        # for multicolumn
  crp=0                 # cell row pointer = cell number in current row
  sideWays=0            # true if sideways encountered in first row
                        #  (back to 0 in later rows)
  isSideways=0          # true if table heading is sideways
  cellAutoStyles=''     # accumulates cell styles for contentAutoStyles
  bibTitle='References' # bibliography title
  bibSet=0              # true if bibTitle changed for a book (with chapters)

  # variables for meta.xml file ---------------------------------------------------
  objN=0
  picN=0
  tabN=0

  settingsMain={ # settings for settings.xml file
        # the values are examples only, to be replaced
        'ViewAreaTop':          '2600',
        'ViewAreaLeft':         '2300',
        'ViewAreaWidth':        '18408',
        'ViewAreaHeight':       '10873',
        'ViewLeft':             '3002',
        'ViewTop':              '4544',
        'VisibleLeft':          '2320',
        'VisibleTop':           '2607',
        'VisibleRight':         '20726',
        'VisibleBottom':        '13478',
        'ZoomType':             '1',
        'ZoomFactor':           '138',
        'BeginLine':            '',     # not understood; sometimes blank
        'EndLine':              '',     # not understood; sometimes blank
        'PrinterSetup':         '',
                                # this is for the Epson 5800
  }

  # variables for styles.xml file -------------------------------------------------
  needsHeaderRow=0      # true if requires table header row for sideways text

  # -------------------------------------------------------------------------------

  # the variables below are set by setIndices() on initialisation so we are free to
  # amend the table without upsetting things elsewhere
  StructP=0             # index of 'Document structure' in objDict
  UnrecP=0              # index of 'Unrecognised' in objDict
  TextP=0               # index of 'Text' in objDict
  TTOCP=0               # index of 'Table of contents' in objDict
  EqnP=0                # index of 'Equation' in objDict
  EqnLP=0               # index of 'Eqn inline' in objDict
  EqnArrNP=0            # index of 'Eqn array numbered' in objDict
  EqnArrP=0             # index of 'Eqn array' in objDict
  ArrP=0                # index of 'Array' in objDict
  EscP=0                # index of 'Escape char' in objDict
  ObjP=0                # index of 'Object parameters' in objDict
  SizeP=0               # index of 'Font size' in objDict
  styleP=0              # index of 'Font style' in objDict
  bibEP=0               # index of 'Bibliography label' in objDict
  refP=0                # index of 'Label' in objDict
  StartObP=0            # index of 'StartObject' in objDict
  EndObP=0              # index of 'EndObject' in objDict
  LabelP=0              # index of '#Label' in objDict
  EqnLabP=0             # index of 'Equation label' in objDict
  FmtP=0                # index of 'Format' in objDict
  VspaceP=0             # index of 'vspace' in objDict
  eqnLab=''             # pending equation label number yet to be inserted
  BlankInserted=0       # toggle for insertion of blank lines

  # ----------------------------------------------------------------------------

  Object=""
  TSp=2                 # running pointer into structure list
  Error="Unknown reason" # error message if aborted
  ErrorLine=" not recorded"
  resultsFile=''

  def setIndices(self):
    "This sets up pointers to various objects."
    self.test1=0
    self.test2=0
    for i,L in self.objDict.iteritems():
      s=L[0]
      if s in self.HoldBlank:
        self.holdBlank.append(i)
      if s == 'Unrecognised':
        self.UnrecP=i
      elif s == 'Document structure':
        self.StructP=i
      elif s == 'Table of contents':
        self.TTOCP=i
      elif s == 'Text':
        self.TextP=i
      elif s == 'Equation':
        self.EqnP=i
      elif s == 'Escape char':
        self.EscP=i
      elif s == 'Object parameters':
        self.ObjP=i
      elif s == 'Font size':
        self.SizeP=i
      elif s == 'Font style':
        self.styleP=i
      elif s == 'Eqn inline':
        self.EqnLP=i
      elif s == 'Bibliography label':
        self.bibEP=i
      elif s == 'Cross reference':
        self.refP=i
      elif s == 'Array':
        self.ArrP=i
      elif s == 'Eqn array numbered':
        self.EqnArrNP=i
      elif s == 'Eqn array':
        self.EqnArrP=i
      elif s == 'StartObject':
        self.StartObP=i
      elif s == 'EndObject':
        self.EndObP=i
      elif s == 'Label':
        self.LabelP=i
      elif s == 'Equation label':
        self.EqnLabP=i
      elif s == 'Format':
        self.FmtP=i
      elif s == 'vspace':
        self.VspaceP=i
    # setIndices

  def setHspaces(self):
    ch=chr(0xc2)+chr(0xa0)
    for i in range(0,200):
      self.hardspaces=self.hardspaces+ch
    # setHspaces

  def findObj(self,s):
    "find index of object from objDict"
    for i,L in self.objDict.iteritems():
      if s == L[0]:
        return i
    # findObj

  def resetIndices(self):
    "reset start indices in objDict"
    for i,L in self.objDict.iteritems():
      L[1]=2
    self.TSp=2
    # resetIndices

  def finalise(self,ok):
    "write remaining xml files and finish"
    th.finishPara()
    makeSettingsMain()
    makeContentMain()
    makeStyles()
    makeMeta()
    makeManifest()
    copyMimetype()
    if (ok):
      removeErrFile()
      s=gv.targetDir+gv.docname
      z=gv.docname+'.zip'
      zipit()                           # zip results into sxw or odt file
      os.chdir(Tmp)
      os.remove(gv.docname+'.ftd')      # clean up ...
      os.remove(gv.docname+'.ftm')
      os.remove(gv.docname+'.ftr')
      rm_dirs(docDir)
      os.chdir(homeDir)
      z='File: '+gv.docname+'.tex successfully converted\nLocated at: '+gv.targetDir
      textGUI(z)
    else:
      reportErr()
    sys.exit(0)
    # finalise

  def extractObj(self):
    "extract next document object"
    try:
      obj=self.objDict[self.StructP][self.TSp]  # get next object type index
    except:
      self.finalise(1)
    i=self.objDict[obj][1]              # get its index pointer
    Object=self.objDict[obj][i]         # get corresponding entry
    self.objDict[obj][1]=i+1            # increment list pointer
    self.TSp=self.TSp+1                 # increment document structure pointer
    i=0
    if obj == self.StartObP or obj == self.EndObP:
      i=obj
    self.lastObject=obj                 # for error reporting
    return obj,i,Object
    # extractObj
  # globalVariables

gv=globalVariables()    # access global variables with gv.
                # must put () to access methods

def textGUI(s):
  "display text in window"
  gv.root = Tix.Tk()
  gv.fsd = ScrolledText(gv.root)
  gv.root.geometry(newGeometry='400x200+300+20')
  gv.fsd.text["wrap"]='word'
  gv.fsd.text["font"]=('Courier','12')
  gv.fsd.pack(side='right')
  gv.fsd.text.insert("end",s)
  gv.fsd.mainloop()

def makeCellStyle(borders,column):
  "make style for a table cell"
  "borders is e.g. 1.cLDTS for table 1 centred left double line and top single line"
  "the columns are handled independently i.e. the same border name can exist in each"
  " column; this is less efficient but eases final declaration of column properties"
  "the sequence must be L, R, T, B when present"
  "column 0 has name Ax, 1 has Bx etc.; x increments as variations in column arise"
  "this allows all A's to be assigned a common width later, etc."
  "the cell styles can be made here but the column styles must be deferred to the end"
  if borders in gv.cellStyles[column]:          # if style already made
    return borders                                      # then return
  cellName='.'+UCletters[column]                # else get letter for column number
  gv.Cnames[column]=gv.Cnames[column]+1         # and next
  cellName=cellName+str(gv.Cnames[column])
  gv.cellStyles[column][borders]=cellName       # enter in dictionary
  s='<style:style style:name="'+gv.tableName+cellName+'" style:family="table-cell">'
  l1='<style:properties '
  l2='fo:background-color="transparent" fo:padding="0.097cm" '
  i=borders.find('.')+2
  if borders[i:] and borders[i] == 'L':
    if borders[i+1] == 'S':
      l2=l2+'fo:border-left="0.002cm solid #000000" '
    else:
      l1=l1+'style:border-line-width-left="0.002cm 0.035cm 0.002cm" '
      l2=l2+'fo:border-left="0.039cm double #000000" '
    i=i+2
  if borders[i:] and borders[i] == 'R':
    if borders[i+1] == 'S':
      l2=l2+'fo:border-right="0.002cm solid #000000" '
    else:
      l1=l1+'style:border-line-width-right="0.002cm 0.035cm 0.002cm" '
      l2=l2+'fo:border-right="0.039cm double #000000" '
    i=i+2
  if borders[i:] and borders[i] == 'T':
    if borders[i+1] == 'S':
      l2=l2+'fo:border-top="0.002cm solid #000000" '
    else:
      l1=l1+'style:border-line-width-top="0.002cm 0.035cm 0.002cm" '
      l2=l2+'fo:border-top="0.039cm double #000000" '
    i=i+2
  if borders[i:] and borders[i] == 'B':
    if borders[i+1] == 'S':
      l2=l2+'fo:border-bottom="0.002cm solid #000000" '
    else:
      l1=l1+'style:border-line-width-bottom="0.002cm 0.035cm 0.002cm" '
      l2=l2+'fo:border-bottom="0.039cm double #000000" '
    i=i+2
  s=s+l1+l2+'></style:properties></style:style>'
  gv.cellAutoStyles=gv.cellAutoStyles+s
  return borders
  # makeCellStyle

def writeTableRow():
  "write next table row"
  "we do the messy conversion of the style-name for sideways headers here rather"
  "than in DOendCell() in case not all header cells are so marked,"
  "as we can only have all-or-none in OpenOffice"
  gv.TableRow=gv.TableRow+'</table:table-row>'
  if gv.sideWays:
    s=gv.TableRow
    i=s.find('text:style-name="')
    j=0
    while i != -1:
      i=i+j+17
      k=i
      j=s[i:].find('"')+i
      s=s[:k]+'Table Heading'+s[j:]
      i=s[j:].find('text:style-name="')
    gv.TableRow='<table:table-header-rows>'+s+'</table:table-header-rows>'
    gv.sideWays=0
    gv.isSideways=1
  if gv.tableRow == gv.tableRows:
    gv.TableRow=gv.TableRow+'</table:table>'
  gv.contentBody.append(gv.TableRow)
  gv.TableRow='<table:table-row>'                       # initialise next row
  gv.crp=0
  gv.rowLines=0
  i=0
  while gv.hlineOveride[i:]:
    gv.hlineOveride[i]=0
    i=i+1
  # writeTableRow

def DOhline(L):
  "handle format item hline for a table"
  gv.rowLines=gv.rowLines+1
  # DOhline

def DOcline(L):
  "handle format item cline for a table"
  "it means add a cell horizontal line (where hline has been omitted)"
  "e.g. \cline {2-2} (or \cline (2-3} ? assume so)"
  L=gv.extractObj()             # get parameter
  s=parseInteger(L[2])-1        # s = zero-based start e.g. 2 above
  e=abs(parseInteger(X[1]))     # e = 1-based end e.g. 2 or 3 above
  while s < e:
    gv.hlineOveride[s]=1
    s=s+1
  # DOcline

def DOhfill(L):
  "handle format item hfill"
  "used by yxlist, enumerate and of general use"
  "fills between margins: actually it pushes text on its right to the right margin"
  "if there is more than one hfill they share the remaining space equally"
  gv.isHfill=1          # space-fill text later
  th.text=th.text+'\\hfill '
  # DOhfill

def DOnoindent(L):
  "handle format item noindent"
  "appear only to be used in quote environment"
  "CHANGES CURRENTLY SET FONT STYLE"
  ts.removeIndentation()                # ensure non-indented font will be used
  return
  # DOnoindent

def DOnonumber(L):
  "handle format item nonumber; ignored;"
  "this is for equation arrays and prevents numbering a line;"
  "we do not label lines in these arrays unless cited, so ignore this item"
  return
  # DOnonumber

def DOtabularnewline(L):
  "handle format item tabularnewline"
  if th.text or gv.TableRow:
    DOendCell(None)                     # then enter cell
  #DOendCell(None)
  writeTableRow()
  gv.tableRow=gv.tableRow+1
  # DOtabularnewline

def DOcentre(L):
  "handle format item centre"
  gv.preJust=ts.setJustify('centre')
  # DOcentre

def DOendcentre(L):
  "end of global centering"
  ts.setJustify(gv.preJust)
  # DOendcentre

def DOleftAlign(L):
  "handle format item leftAlign"
  gv.preJust=ts.setJustify('start')
  # DOleftAlign

def DOendleftAlign(L):
  "end of leftAlign"
  ts.setJustify(gv.preJust)
  # DOendleftAlign

def DOrightAlign(L):
  "handle format item rightAlign"
  gv.preJust=ts.setJustify('end')
  # DOrightAlign

def DOendrightAlign(L):
  "end of rightAlign"
  ts.setJustify(gv.preJust)
  # DOendrightAlign

bl={'S': 1,'D': 2}

def setMulti(cf,cellsSpanned):
  "replace default cell style with multiformat one"
  "the trick is to look at the left border of the right neighbouring cell"
  "and only add border lines if more are wanted than already exist"
  "if there is no right neighbour, then compare with right border of last cell"
  rnlbl=0                               # right neighbour left border lines default
  lcrb=0                                # last cell right border
  i=gv.crp+cellsSpanned                 # column number of right neighbour
  if gv.cellRow[i:]:                    # if right neighbour exists
    L=gv.cellRow[i]                             # then get its
    i=L.find('L')                               # left border style
    if i != -1:
      rnlbl=bl[L[i+1]]                          # rnlbl = 1 or 2 if exists
  else:                                 # else
    L=gv.cellRow[-1]
    j=L.find('R')                               # get right border style of last cell
    if j != -1:
      lcrb=bl[L[j+1]]                           # lcrb = 1 or 2 if exists
  L=parseParam(gv.mergeformat[1:-1])    # parse cell params into L
  i=cf.find('.')+1
  if L[2] != 'p':                       # ignore p
    cf=cf[:i]+L[2]+cf[i+1:]             # else replace justification (c l or r)
  i=L[0]                                # i = number of left lines
  s=L[1]                                # s = remaining part
  j=cf.find('L')
  if i>0 and j != -1:                   # if both require left border
    j=j+1
    if i == 2 and cf[j] == 'S':                 # then multi double overides
      cf=cf[:j]+'D'+cf[j+1:]
    elif i ==1 and cf[j] == 'D':                # or multi single overides
      cf=cf[:j]+'S'+cf[j+1:]
    j=j-1
  k=cf.find('R')                        # find index of current format 'R'
  ebl=0
  while s[ebl:] and s[ebl] == '|':
    ebl=ebl+1                           # ebl is number of right '|' chars for multi
  ebl=ebl-rnlbl                         # number of extra right border lines for multi
  if k != -1:
    if ebl>0:                           # if both required right border
      k=k+1
      if ebl == 2 and cf[k] == 'S':             # then multi double overides
        cf=cf[:k]+'D'+cf[k+1:]
      elif ebl == 1 and cf[k] == 'D':           # or multi single overides
        cf=cf[:k]+'S'+cf[k+1:]
  elif ebl>0:                           # else if multi requires right border
    if ebl == 1:
      ch='RS'
    else:
      ch='RD'
    if j == -1:                                 # if no 'L' in current format
      j=cf.find('.')+1                               # then set j to skip e.g. '.c'
    cf=cf[:j+2]+ch+cf[j+2:]             # splice in required format
  elif lcrb:                            # else if last cell has right border
    if lcrb == 1:
      ch='RS'
    else:
      ch='RD'
    if j == -1:
      j=cf.find('.')+1                          # then set j to skip e.g. '.c'
    cf=cf[:j+2]+ch+cf[j+2:]             # then use it
  return cf
  # setMulti

def DOendCell(L):
  "handle format item endCell"
  "we invented this to replace '&' for table and array cell delimiters"
  "the current cell content is in gv.contentBody[gv.cellStart:]"
  if ObjContext.state[:3] == 'eqn':             # comes here if in an array
    if th.text.strip()[-1:] == '#' or not th.text:
      th.text=th.text+phTerm
    th.text=th.text+'#'
    gv.arrCol=gv.arrCol+1
  if ObjContext.state == 'table':
    cellInc=1
    cf=gv.cellRow[gv.crp]               # get next cell format e.g. 1.cLSTS
    if gv.rowLines == 1 or gv.hlineOveride[gv.crp]:
      cf=cf+'TS'
    elif gv.rowLines == 2:
      cf=cf+'TD'
    if gv.tableRow == gv.tableRows:     # if last row
      cf=cf+'BS'                                # then include single bottom line
    if gv.mergecols:                    # if is multicolumn
      cellInc=gv.mergecols
      cf=setMulti(cf,cellInc)                   # then overide default cell props
    csn=makeCellStyle(cf,gv.crp)        # make cell style and get cell style name
    if th.currentPara:                  # this allows format changes to work
      th.text=th.extractSpan()                  # normally into th.text
    i=csn.find('.')
    ts.setJustify(cellJustify[csn[i+1]])        # csn[i+1]= c,l or r
    ts.setPStyle()                              # make P style for cell
    csn=gv.cellStyles[gv.crp][csn]              # get style name for cell
    s='<table:table-cell table:style-name="'+gv.tableName+csn+'" '
    if gv.mergecols:
      s=s+'table:number-columns-spanned="'+str(cellInc)+'" '
      if gv.lineLen:
        gv.lineLen = -gv.lineLen                # lineLen<0 indicates multi-column
      else:
        gv.lineLen = - 0.1
    s=s+'table:value-type="string"><text:p text:style-name="'+ts.makeStyleName('P')+'">'
    if th.text[-9:] == '</text:p>':
      s=s+th.text+'</table:table-cell>'
    else:
      s=s+th.text+'</text:p></table:table-cell>'
    gv.colWids[gv.crp].append(gv.lineLen)       # store in column widths
    gv.TableRow=gv.TableRow+s
    while cellInc>1:
      gv.crp=gv.crp+1
      gv.colWids[gv.crp].append(-100)           # put -100 in suppressed columns
      cellInc=cellInc-1
    gv.crp=gv.crp+1
    initCell()
  # DOendCell

def DOpagestyle(L):
  "page style is not yet implemented"
  gv.extractObj()               # discard style
  # DOpagestyle

def handleFormat(L):
  "handle the various special format items such as hfill etc."
  exec('s=DO'+L[2]+'(L)')                       # handle format item
  return s
  # handleFormat

def enterContents(diry,centred):
  "prepare entry in body of main content.xml for centred equation"
  "fr1 is predefined"
  if not centred:
    ts.setTStyle()
    f='<text:span text:style-name="'+ts.makeStyleName('T')+'">'
  else:
    if gv.isLabelled:
      ts.setJustify('end')
      padding=max(0,(gv.pageWidth-gv.eqnWidth)/2-gv.labelWidth)
        # for right justified paragraph, padding width is margin less label size
    elif centred == 2:                  # if numbered equation array
      ts.setJustify('end')
    elif centred:
      ts.setJustify('centre')
    ts.setPStyle()
    f='<text:p text:style-name="'+ts.makeStyleName('P')+'">'
  s='<draw:object draw:style-name="fr1" draw:name="Object'
  s=s+str(gv.objN)+'" text:anchor-type="as-char" svg:x="0cm" '
  s=s+' svg:width="'+r(gv.eqnWidth)+'cm" svg:height="'
  s=s+r(gv.eqnHeight)+'cm" '
  s=s+'draw:z-index="'+str(gv.eqnZI)+'" xlink:href="#./'+diry+'" xlink:type="simple"'
  s=s+' xlink:show="embed" xlink:actuate="onLoad"/>'
  if gv.isLabelled:
    n=gv.charWid[32]*ts.defFontSize/12.0
    n=int(padding*1000.0/n+0.5)
    s=s+gv.hardspaces[:2*n]+gv.EqnLab
    #s=s+'<text:s text:c="'+str(n)[:-2]+'"/>'+gv.EqnLab
        # [:-2] removes the .0
  if centred:
    gv.contentBody.append(f+s+'</text:p>')
  else:
    th.finishSpan()
    th.currentPara=th.currentPara+f+s+'</text:span>'
  # enterContents

def handleError(L):
  "called if invalid object encountered"
  s='Invalid object encountered\n'
  s=s+'Object number = '+str(gv.objDict[gv.StructP][2]-1)
  s=s+'\nObject type   = '+gv.objDict[gv.lastObject][0]
  makeErrorFile(s)
  gv.finalise(0)
  # handleError

def saveContext(major):
  "actions required before handling a compound object"
  "if it is not classed as major then the text it extracts must be combined"
  "with the existing text in restoreContext()"
  gv.textStack.append(th.text)
  gv.textStack.append(gv.isHfill)
  th.text=''
  ts.isHfill=0
  gv.textStack.append(major)
  ts.saveStyle()
  #gv.textStack.append(ts.savedStyle)
  # saveContext

def restoreContext():
  "restore context on conclusion of compound object handling"
  #ts.savedStyle=gv.textStack.pop()
  ts.restoreStyle()
  major=gv.textStack.pop()
  gv.isHfill=gv.textStack.pop()
  if major:
    th.text=gv.textStack.pop()
  else:
    s=gv.textStack.pop()
    if len(s)>0 and len(th.text)>0:
      if s[-1] != ' ' and s[0] != ' ':
        s=s+' '
    th.text=s+th.text
  # restoreContext

"       policy for handling sequence of objects"
""" ------------------------------------------------------------------------------
    the complete font style is saved by handleStart() and restored by handleEnd()
    the current text is accumulated in th.text
        this is done either in
          main Text
          or in objects with the current compound object name
                (e.g. Title, Abstract etc.)
    the current text is written into gv.contentBody when:
        1. a font change occurs; in the middle of a line this requires a
           <text:span ...> ... <text:span/> bracket combined with a "Tx" font type
           rather than a "Px" one
        2. at the end of the current compound object
        3. when '\\newline' is encountered
        4. when another escape char is encountered, as that may imply a font change

-------------------------------------------------------------------------------"""

def handleStart(L):
  "The entry for a StartObject is the object name"
  "we extract its handling function using the cross-reference for specFunc"
  M=gv.xRef[L[2]]               # get specFunc items from object name in L[2]
  if M[1]<2:                    # M = [<object name>,<parameter>]
    th.finishPara()             # ensure possible outstanding paragraph is handled
    handleBlankLine('')         # insert blank line
    saveContext(M[1])
  elif M[1] == 2:
    saveContext(0)
  exec(M[0]+'(L)')              # M[0] is name of major special handling function
  # handleStart

def handleEnd(L):
  "handle EndObject when it should be followed by a blank line"
  "otherwise call restoreContext() directly"
  th.finishPara()
  handleBlankLine('')           # insert blank line (before restore style for 'keep')
  try:
    M=gv.xRef[L[2]]
    if M[1]<3:
      restoreContext()
  except:
    restoreContext()
  if ObjContext.state == 'other':
    ObjContext.unset()
  # handleEnd

def handleObjBody(L):
  if ObjContext.state == 'standard':
    ObjContext.set('other')
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    handleObject(L)             # endObject is also handled
  # handleObjBody

def handleElement(L):
  "this is for cases where no special processing is required"
  th.text=th.text+L[2]
  # handleElement

def handleElSize(L):
  "determine length of text"
  "one line may consist of text fragments between e.g. font changes"
  gv.lineLen=gv.lineLen+ts.fontScale*textLength(L[2]) # N.B. is cumulative across fonts
  th.text=th.text+L[2]
  # handleElSize

def Abstract(L):
  "do abstract"
  ObjContext.set('abstract')
  ts.setJustify('centre')
  ts.setWeight('bold')
  th.text='Abstract'
  th.finishPara()
  ts.setWeight('normal')
  ts.setMargin(1)
  ts.setSize(-2)                # smaller size for abstract
  handleObjBody(L)
  ObjContext.unset()
  # Abstract

def handleTOC(L):
  "so far we have not added page numbers to this"
  ObjContext.set('toc')
  saveContext(1)
  ts.setSize(2)
  ts.setWeight('bold')
  th.text='Contents'
  th.finishPara()
  ts.setSize(-1)
  ts.setWeight('normal')
  ts.setMargin(1)
  i=2
  while gv.objDict[gv.TTOCP][i:]:
    s=gv.objDict[gv.TTOCP][i]
    j=0
    while s[j] == ' ':
      j=j+1
    ts.setIndent(4*j)
    th.textState=''
    th.text=' '
    th.enterText(1)             # essential to prevent persistent LucidaBright
    th.text=s[j:]
    th.finishPara()
    i=i+1
  ts.setSize(0)
  handleEnd(L)
  ObjContext.unset()
  # handleTOC

def Bibliog(L):
  "Bibliography"
  "the last entry is written by handleEnd()"
  ObjContext.set('bibl')
  ts.saveStyle()
  ts.setSize(4)                         # make size = def size + 4
  ts.setMargin(1)
  ts.setWeight('bold')
  th.text=gv.bibTitle
  th.finishPara()
  ts.restoreStyle()
  gv.extractObj()                       # skip parameter
  handleObjBody(L)                      # object parameter skipped (of unknown use)
  ObjContext.unset()
  # Bibliog

def BibLabel(L):
  "Bibliography label; comes before bibliography text"
  L=gv.extractObj()                     # get first object
  if L[0] == gv.ObjP:                   # if is parameter
    gv.extractObj()                     # then skip it and skip Lyx label
  gv.extractObj()                       # skip end object
  if gv.bibLab>0:                       # if not first entry
    th.finishPara()                     # then write previous one
  gv.bibLab=gv.bibLab+1
  th.text='['+str(gv.bibLab)+'] '       # construct required label
  # BibLabel

def Caption(L):
  "handle specially to remove blankline"
  ObjContext.set('caption')
  ts.setJustify('centre')
  ts.setPStyle()                                # make P style for cell
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    handleObject(L)
  ObjContext.unset()
  # Caption

def handleCaption(L):
  "this is part of a figure, perhaps including a label"
  gv.picNum=gv.picNum+1
  th.text='Figure '+str(gv.picNum)+': '+L[2]
  gv.capLen=textLength(th.text)*ts.fontScale
  # handleCaption

def handleCitation(L):
  "handle citation; problem is text line has already been entered"
  s=L[2]
  if gv.wasExpanded:
    s=s[:-1]+', '+gv.wasExpanded[1:]    # strip leading '[' from gv.wasExpanded
    gv.wasExpanded=''
  S=gv.contentBody.pop()                # recover previous text line
  T=gv.contentBody.pop()                # recover previous text line to that
  i=T.rfind('<')
  T=T[:i]+' '+s+T[i:]                   # splice-in citation
  gv.contentBody.append(T)              # restore all to
  gv.contentBody.append(S)              # content
  # handleCitation

def Title(L):
  "handle directly to insert centering"
  ts.setJustify('centre')
  ts.setSize(8)
  ts.setStyle('sserif')
  gv.extractObj()                       # skip size change
  gv.extractObj()                       # skip font change
  handleObjBody(L)
  gv.extractObj()                       # skip size change
  gv.extractObj()                       # skip font change
  # Title

def Author(L):
  "handle specially to set font"
  ts.setJustify('centre')
  ts.setSize(2)
  handleObjBody(L)
  # Author

def Date(L):
  "handle specially to set font"
  ts.setJustify('centre')
  ts.setSize(2)
  handleObjBody(L)
  # Date

def Part(L):
  "start of new document Part"
  handleBlankLine('')
  ts.setSize(+6)
  ts.setWeight('bold')
  ts.setKeep('keep')
  th.text='part'
  handleObjBody(L)
  # Part

def handlePart(L):
  "handle directly to permit font changes"
  if th.text == 'part':
    th.text=L[2]+gv.hardspaces[:6]
  else:
    th.text=th.text+L[2]
  # handlePart

def Chapter(L):
  ObjContext.set('chap')
  if not gv.bibSet:
    gv.bibTitle='Bibliography'
    gv.bibSet=1
  ts.setSize(+4)
  ts.setWeight('bold')
  ts.setKeep('keep')
  L=gv.extractObj()             # get Chapter #
  handleObject(L)               # write it
  gv.BlankInserted=0
  handleBlankLine('')
  ts.setSize(+8)
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    handleObject(L)             # endObject is also handled
  ObjContext.unset()
  # Chapter

def Section(L):
  ts.setSize(+2)
  ts.setWeight('bold')
  ts.setKeep('keep')
  handleObjBody(L)
  # Section

def SubSection(L):
  ts.setSize(+1)
  ts.setWeight('bold')
  ts.setKeep('keep')
  handleObjBody(L)
  # SubSection

def SubSubSection(L):
  ts.setWeight('bold')
  ts.setKeep('keep')
  handleObjBody(L)
  # SubSubSection

def Comment(L):
  "ignore, was only for Lyx document and should have been omitted"
  while L[0] != gv.EndObP:
    L=gv.extractObj()
  # Comment

def enterHex(sr):
  "parse hex number and enter it"
  ch=''                 # then convert it
  while sr[0:2] == '0x':                        # handle multiple char codes
        ch=ch+chr(eval(sr[:4])) # (ensure always of form 0xnn in dictionary)
        sr=sr[1:]
        j=sr.find('0x')
        sr=sr[j:]                               # if j = -1, loop is terminated
  if ObjContext.state[:3] == 'eqn':
        th.text=th.text+ch+sr
  else:
        th.finishSpan()                         # close any outstanding span
        ts.saveStyle()
        ts.setFont('symbol')                    # set font for symbol
        th.text=ch
        th.enterText(1)                         # write completed escape sequence
        th.finishSpan()
        ts.restoreStyle()
  # enterHex

def parseInteger(s):
  "parse ascii integer from s, only returning value"
  i=0
  while not s[i].isdigit():
    i=i+1
  j=i
  while s[i].isdigit():
    i=i+1
  if j and s[j-1] == '-':
    j=j-1
  return eval(s[j:i])
  # parseInteger

def parseInt(s):
  "parse ascii integer from s"
  i=0
  while not s[i].isdigit():
    i=i+1
  j=i
  while s[i].isdigit():
    i=i+1
  if j and s[j-1] == '-':
    j=j-1
  return eval(s[j:i]),s[i:]
  # parseInt

def parseReal(s):
  "parse ascii real number from s"
  i=0
  while not s[i].isdigit() and not s[i] == '.':
    i=i+1
  j=i
  while s[i].isdigit() or s[i] == '.':
    i=i+1
  if j and s[j-1] == '-':
    j=j-1
  return eval(s[j:i]),s[i:]
  # parseReal

def handleEsc(L):
  "handle escape characters"
  if L[2] == '\\newline':
    if ObjContext.state[:3] == 'eqn':
      if ObjContext.state == 'eqna':
        if not th.text or th.text.rstrip()[-1] == '#':
          th.text=th.text+phTerm                # first add
        while gv.arrCol<gv.arrCols-1:
          th.text=th.text+'#'+phTerm    # phantom terms as necessaey
          gv.arrCol=gv.arrCol+1
        th.text=th.text+'##'            # row separator for an array
        gv.arrCol=0
    else:
      th.finishPara()
      if ObjContext.state in ['verse','quote','lyxlist','list','descr','table']:
        handleBlankLine('')
        if ObjContext.state in ['lyxlist','list','descr','table']:
          ts.setIndent(0)               # for embedded paras in lists
  elif ObjContext.state == 'lyxcode' and L[2] == '\\hardspace' and th.text:
    th.text=th.text+' '         # else wordprocessor wraps line in middle of word
                                # but retain hardspace at beginning of a line
  else:
    if L[2][:6] == '\\latin':   # if is latin char
      x=ord(L[2][6])                    # then get its ordinal value
      if x<0xc0:                        # and if < 0xc0
        sr='{0xc2'+hex(x)+'}'           # then code directly preceded by 0xc2
      else:
        sr='{0xc3'+hex(x-0x40)+'}'      # else subtract 0x40 and precede by 0xc3
    else:
      sr=gv.escapeDict[L[2]]    # else sr = replacement string from dictionary
    ch=sr
    if sr[0] == '@':            # if abbreviation control char
      return                            # ignore it
    j=sr.find('0x')
    if sr[0] == '{' and ObjContext.state[:3] != 'eqn':
      sr=sr[1:]
        # this arose because we had to enclose hex codes in {} for StarMath
        # but we don't want '{' being printed
      j=sr.find('0x')
    if j != -1:                 # if char code supplied e.g. for a bullet
      if j:
        if sr[:4] == 'size':            # e.g. size 2 {#...] in equation or
          ch=parseInteger(sr[5:])
          ch=ts.defFontSize+ch
          th.text=th.text+' size '+str(ch)+ '{'
          j=sr.find('0x')
        else:
          th.text=th.text+' '+sr[:j]
        sr=sr[j:]
      enterHex(sr)
    else:
      if ObjContext.state[:3] == 'eqn':
        ch=' '+ch+' '                           # ensure keywords are space-
                                                # separated (spaces are ignored
                                                # by OpenOffice for StarMath)
      else:
        if ch in nonEqnEsc:                     # if e.g. dotslow
          ch=nonEqnEsc[ch]                      # then replace
          if ch[0] == '{':                      # but if is symbol
            enterHex(ch[1:])                    # then handle it
            ch=''                               # and avoid entering directly
      th.text=th.text+ch
  # handleEsc

def Figure(L):
  "handle figure float"
  "is not floated, but put in a paragraph-anchored frame to keep graphic and"
  "caption together"
  ObjContext.set('fig')
  Len=len(gv.contentBody)       # note current length of content body
  z=gv.eqnZI+1
  gv.eqnZI=z                    # this must precede that of the graphic
  gv.lineLen=0
  extraLines=0
  gv.capLen=0
  w=0
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    handleObject(L)             # endObject is also handled
    if gv.lineLen:
      w=max(gv.lineLen,w)
      gv.lineLen=0
      extraLines=extraLines+1
  temp=[]
  while len(gv.contentBody)>Len:
    temp.append(gv.contentBody.pop())
  gv.frameN=gv.frameN+1
  w=max(gv.pWid,gv.capLen,w)
  if gv.capLen:
    gv.pHt=gv.pHt+1
  gv.pHt=gv.pHt+extraLines
  s=textPara+ts.makeStyleName('P')+'">'
  s=s+'<draw:text-box draw:style-name="fr3" draw:name="frame'+str(gv.frameN)+'" '
  s=s+'text:anchor-type="paragraph" '
  s=s+'svg:width="'+r(w)+'cm" svg:height="-1cm" '
  s=s+'draw:z-index="'+str(z)+'">'
  gv.contentBody.append(s)
  while temp:
    gv.contentBody.append(temp.pop())
  gv.contentBody.append('</draw:text-box></text:p>')
  ObjContext.unset()
  # Figure

def Graphics(L):
  "handle graphics object in fixed position, unless is floated in Figure"
  gv.picN=gv.picN+1             # increment number of objects
  ObjContext.set('fig')         # set context to control other procedures
  ts.setJustify('centre')
  ts.setPStyle()
  handleObjBody(L)
  i=gv.picName.rfind('.')+1     # index of picture type
  s='<manifest:file-entry manifest:media-type="image'+xmlSep
  s=s+gv.picName[i:]+'" manifest:full-path="Pictures'+xmlSep+gv.picName+'"/>'
  gv.picManifest=gv.picManifest+s
  th.paraCount=0
  ObjContext.unset()
  # Graphics

def getVal(s):
  "convert s to binary"
  "s[0] is least significant byte"
  x=0
  i=len(s)
  while i:
    x=256*x+ord(s[i-1])
    i=i-1
  return x
  # getVal

def getInt(s):
  "convert s to binary"
  "s[0] is most significant byte"
  x=0
  i=len(s)
  j=0
  while j<i:
    x=256*x+ord(s[j])
    j=j+1
  return x
  # getInt


def bmpSize(pn):
  "return size of bmp image"
  ff=open(pn,'rb')
  s=ff.read(128)
  w=getVal(s[18:22])
  h=getVal(s[22:26])
  ff.close()
  return w,h
  # bmpSize

def pcxSize(pn):
  "return size of pcx image"
  ff=open(pn,'rb')
  s=ff.read(128)
  w=getVal(s[8:10])-getVal(s[4:6])+1
  h=getVal(s[10:12])-getVal(s[6:8])+1
  ff.close()
  return w,h
  # pcxSize

def gifSize(pn):
  "return size of gif image"
  ff=open(pn,'rb')
  s=ff.read(128)
  w=getVal(s[6:8])
  h=getVal(s[8:10])
  ff.close()
  return w,h
  # gifSize

def pngSize(pn):
  "return size of png image"
  ff=open(pn,'rb')
  s=ff.read(128)
  ihdr=s.find('IHDR')+4
  w=getInt(s[ihdr:ihdr+4])
  h=getInt(s[ihdr+4:ihdr+8])
  ff.close()
  return w,h
  # pngSize

def jpgSize(pn):
  "return size of jpg image"
  "the bytes 0xff 0xc0 0x00 0x11 0x08 precede the height and width"
  "such blocks may occur more than once, so we find the last one"
  ff=open(pn,'rb')
  s=ff.read(20000)
  c=chr(0xff)
  c=c+chr(0xc0)
  c=c+chr(0x00)
  c=c+chr(0x11)
  c=c+chr(0x08)
  i=0
  j=s.find(c)
  while j != -1:
    i=i+j+5
    j=s[i:].find(c)
  w=getInt(s[i+2:i+4])
  h=getInt(s[i:i+2])
  ff.close()
  return w,h
  # jpgSize

def epsSize(pn):
  "return size of eps image"
  ff=open(pn,'rb')
  s=ff.read(1000)
  i=s.find('BoundingBox')+12
  L=s[i:]
  L=parseInt(L)
  w=L[0]
  L=parseInt(L[1])
  h=L[0]
  L=parseInt(L[1])
  w=L[0]-w
  L=parseInt(L[1])
  h=L[0]-h
  ff.close()
  return w,h
  # epsSize

def psSize(pn):
  "return size of Postscript image"
  ff=open(pn,'rb')
  s=ff.read(2000)
  i=s.find('Transformation matrix')+21
  s=s[i:]
  L=parseInt(s)
  w=L[0]
  L=parseInt(L[1])
  L=parseInt(L[1])
  L=parseInt(L[1])
  h=L[0]
  ff.close()
  return w,h
  # psSize

def xbmSize(pn):
  "return size of xbm image"
  ff=open(pn,'rb')
  s=ff.read(128)
  i=s.find('_width')+6
  L=s[i:]
  L=parseInt(L)
  w=L[0]
  i=s.find('_height')+7
  L=s[i:]
  L=parseInt(L)
  h=L[0]
  ff.close()
  return w,h
  # xbmSize

def calcPicSize(picName):
  "find pic size from its name"
  #L=rgbimg.sizeofimage(picName)        fails miserably
  i=picName.rfind('.')+1
  s=picName[i:].lower()
  try:
    exec('L='+s+'Size(picName)')        # calc pic size if recognised type
  except:
    L=[picScale*12,picScale*9]          # else default 12cm x 9cm image
  gv.pWid=1.0*L[0]/picScale
  gv.pHt=1.0*L[1]/picScale
  if gv.pWid>gv.pageWidth:
    f=1.0*gv.pageWidth/gv.pWid
    gv.pWid=gv.pageWidth
    gv.pHt=gv.pHt*f
  # calcPicSize

def handleGraphics(L):
  "this is the picture name"
  subDir='Pictures'
  if not gv.picDirMade:                 # if Picture sub dir not made
    os.mkdir(docDir+subDir)             # then make it
    gv.picDirMade=1
  i=L[2].rfind(dirySep)
  if i != -1:
    fname=L[2][i:]
  else:
    fname=L[2]
  if L[2][0] != dirySep:                # if relative path given
    path=picDir+L[2]                    # then assume is default path
  else:
    path=L[2]
  while fname[0] == dirySep:
    fname=fname[1:]
  calcPicSize(path)
  ff=open(path,'rb')
  fw=open(docDir+subDir+dirySep+fname,'wb')
  s=ff.read()
  fw.write(s)
  ff.close()
  fw.close()
  gv.graphicN=gv.graphicN+1
  gv.eqnZI=gv.eqnZI+1
  s=textPara+ts.makeStyleName('P')+'">'
  s=s+'<draw:image draw:style-name="fr2" draw:name="Graphic'+str(gv.graphicN)+'" '
  s=s+'text:anchor-type="as-char" '
  s=s+'svg:width="'+r(gv.pWid)+'cm" svg:height="'+r(gv.pHt)+'cm" '
  s=s+'draw:z-index="'+str(gv.eqnZI)+'" '
  s=s+'xlink:href="#Pictures'+xmlSep+fname+'" xlink:type="simple" xlink:show="embed" '
  s=s+'xlink:actuate="onLoad"'+xmlSep+'></text:p>'
  gv.picName=fname
  gv.contentBody.append(s)
  # handleGraphics

def handleLabel(L):
  "ignore all except Figure labels"
  "others have been converted and are in Cross Reference"
  if ObjContext.state == 'fig':
    th.text=th.text+L[2]
  # handleLabel

def RightAddr(L):
  "right-justified address"
  ObjContext.set('rightad')
        # this is necessary to stop extra blank lines following \newline
  gv.lineLen=0
  gv.raStart=len(gv.contentBody)  # start of right address entries in gv.contentBody
  maxlen=0
  more=1
  indnt=ts.getIndent()
  ts.setIndent(0)
  while more:                           # scan whole object, calculating lengthd
    L=gv.extractObj()
    if L[0] == gv.EscP:
      if L[2] == '\\newline':
        if gv.lineLen > maxlen:
          maxlen=gv.lineLen             # find max line length
        gv.lineLen=0
    elif L[0] == gv.EndObP:
      more=more-1
    s=L[2]
    s=s.strip()                         # remove all redundant white spaces
    M=[L[0],L[1],s]                     # this is because python will not do L[2]=...
    handleObject(M)                     # endObject is also handled
  if gv.lineLen>maxlen:
    maxlen=gv.lineLen
  i=ts.getMargin()
  if i:
    maxlen=maxlen+ts.margins[i]
  cw=(gv.spaceW+1)*ts.fontScale         # width of one space
  sp=max(0,int((gv.pageWidth-maxlen)/cw)) # spaces to pad each line with
  i=gv.raStart
  k=0
  while gv.contentBody[i:]:             # search new content for guessed margins
    s=gv.contentBody[i]
    j=s.find('<text:p ')
    k=s[j:].find('>')                   # find end of '<text:p ...>'
    if k != -1:
      j=j+k+1
      s=s[:j]+gv.hardspaces[:2*sp]+s[j:]        # prepend line with space filler
      #s=s[:j]+'<text:s text:c="'+str(sp)+'"/>'+s[j:]   # prepend line with space filler
    gv.contentBody[i]=s
    i=i+1
  ts.setIndent(indnt)
  ObjContext.unset()
  # RightAddr

def Quotation(L):
  "all paragraphs have a margin , so set style accordingly"
  ObjContext.set('quotation')   # set context to control other procedures
        # this is especially important to suppress extra blank lines when is nested
  ts.incrMargin(4)
  ts.setIndent(5)
  handleObjBody(L)
  ts.reduceMargin(2)
  ObjContext.unset()
  # Quotation

def handleQuotation(L):
  "all paragraphs have a margin"
  "the first line of the first paragraph is indented"
  "the parser left '\n\n' to separate paragraphs as the Tex document adds"
  "unwanted '\n' chars in the middle of paras !"
  s=L[2]
  i=s.find('\n')
  while i != -1:                        # while para contains '\n'
    if s[i+1:] and s[i+1] == '\n':      # then if is '\n\n'
      th.text=th.text+s[:i]             # then include text up to '\n\n'
      th.finishPara()                   # and write it (th.text set to '')
      s=s[i+1:]                         # set s to rest of para
    else:
      s=s[:i]+' '+s[i+1:]       # else splice '\n' out
    i=s.find('\n')
  th.text=th.text+s
  # handleQuotation

def Quote(L):
  "all paragraphs have a margin, so set style accordingly"
  ts.incrMargin(4)
  ObjContext.set('quote')       # set context to control other procedures
  handleObjBody(L)
  ts.reduceMargin(2)
  ObjContext.unset()
  # Quote

def handleQuote(L):
  "all paragraphs have a margin, and paras are separated by blank lines"
  "the parser left '\n\n' to separate paragraphs as the Tex document adds"
  "unwanted '\n' chars in the middle of paras !"
  s=L[2]
  th.text=th.text+s
  # handleQuote

def handleText(L):
  "text items; remove redundant '\n' chars placed by LaTex"
  s=L[2]
  i=s.find('\n')
  while i != -1:                # while para contains '\n'
    s=s[:i]+' '+s[i+1:]         # splice '\n' out
    i=s.find('\n')
  th.text=th.text+s
  # handleText

def Verse(L):
  "verse  has a left margin and a ragged right margin"
  ObjContext.set('verse')       # set context to control other procedures
  ts.setJustify('start')
  ts.incrMargin(4)
  ts.setIndent(-5)
  handleObjBody(L)
  ObjContext.unset()
  # ragged justification is removed by handleEnd()
  # Verse

def handleVerse(L):
  th.text=th.text+L[2]
  #  handleVerse

def Lyxcode(L):
  "set up lyxcode environment"
  ObjContext.set('lyxcode')
  ts.setStyle('ttype')
  ts.incrMargin(4)
  ts.setIndent(0)
  handleObjBody(L)
  ObjContext.unset()
  # Lyxcode

def insetObject(L):
  "do object with margin"
  ObjContext.set('list')
  ts.incrMargin(4)
  ts.setIndent(-3)
  ts.setPStyle()                # make P style for cell
  handleObjBody(L)
  ObjContext.unset()
  # insetObject

def padText(cm):
  "pad label text"
  s=th.currentPara+th.text
  if 'text:s text:c=' in s:             # if hfills done
    return                              # then abort
  n=th.calcSize(s)
  if n<cm:
    i=int((cm-n)/(gv.spaceW*ts.fontScale)-0.5)
  else:
    i=2
  th.text=th.text+gv.hardspaces[:2*i]
  # padText

def Description(L):
  "description environment"
  ts.incrMargin(2)
  ObjContext.set('descr')
  handleObjBody(L)
  ObjContext.unset()
  # Description

def LyxList(L):
  "handle Lyx-style list"
  "the label width is set to the nearest multiple of 0.25cm to that defined by"
  "the parameter; this preserves the integrity of currentStyle; if we set a margin"
  "that is not a multiple of 0.25 then we risk the same style name being used with"
  "different actual margin values as only the number of offsets is stored in"
  "currentStyle;"
  "the negative indent is set to the label width"
  ObjContext.set('lyxlist')
  gv.labelWid.append(gv.labelW)         # put current params on stack
  gv.labelWid.append(gv.labelN)
  gv.labelWid.append(gv.lyxIndent)
  L=gv.extractObj()                     # get parameters
  L=L[2].strip('{}')
  i=textLength(L)*ts.fontScale          # calculate label width from parameters
  gv.labelN=int(i/ts.margins[1]+0.5)    # labelN is number of 0.25cm margin increments
  gv.labelW=gv.labelN*ts.margins[1]     # labelW is nearest multiple of 0.25cm
  gv.lyxIndent=-int(10*gv.labelW)       # label indent in mm
  ts.incrMargin(gv.labelN)
  handleObjBody(L)                      # handle the list
  ObjContext.unset()
  gv.lyxIndent=gv.labelWid.pop()
  gv.labelN=gv.labelWid.pop()
  gv.labelW=gv.labelWid.pop()
  # LyxList

def handleListItem(L):
  "handle list item element"
  s=L[2].lstrip()
  if not s:
    th.text=th.text+L[2]
    return
  elif s[0] == '#':     # bullet required
    handleEsc(B)
  elif s[0] == '[' and ObjContext.state == 'lyxlist':   # start of Lyx-list item
    ts.setIndent(gv.lyxIndent)
    i=L[2].find('[')
    j=L[2].find(']')
    if j != -1:
      th.text=L[2][i+1:j]       #+'<text:tab-stop/>'
      padText(gv.labelW)        # pad to label width
      th.enterText(1)
      th.text=L[2][j+1:]
      th.enterText(1)
    else:
      th.text=L[2][i+1:]
      gv.LLbrPend=1
  elif s[0] == '[' and ObjContext.state =='descr':      # start of description item
    ts.setIndent(-5)
    i=L[2].find('[')
    j=L[2].find(']')
    ts.saveStyle()
    ts.setWeight('bold')
    if j != -1:
      th.text=L[2][i+1:j]+gv.hardspaces[:4]
      th.enterText(1)
      ts.restoreStyle()
      th.text=L[2][j+1:]
      th.enterText(1)
    else:
      th.text=L[2][i+1:]
      th.enterText(1)
      gv.LLbrPend=1
  elif ']' in L[2] and gv.LLbrPend: # end of label of current Lyx-list or descr item
    i=L[2].find(']')
    th.text=th.text+L[2][:i]
    if gv.isHfill:                      # if hfills present
      th.text=th.doHfill(gv.labelW,th.text)  # then assume tab width & handle them
    if th.text and th.text[0] == '#':   # if "mixed nesting"
      ts.setIndent(gv.lyxIndent)        # was -9
      th.text=th.text+gv.hardspaces[:4]
    else:
      if ObjContext.state =='lyxlist':
        padText(gv.labelW)              # pad to label width
      else:
        th.text=th.text+gv.hardspaces[:4]
    th.enterText(0)
    if ObjContext.state =='descr':
      ts.restoreStyle()
    th.text=L[2][i+1:]
    th.enterText(1)
    gv.LLbrPend=0
  elif not th.currentPara and ObjContext.state == 'list':
    ts.setIndent(-3)
    th.text=th.text+L[2]
  else:
    th.text=th.text+L[2]
  # handleListItem

def URL(L):
  "handle URL, as its tag has been stored as a parameter"
  ObjContext.set('url')
  ts.saveStyle()
  L=gv.extractObj()             # get parameter
  s=L[2]
  i=s.find('[')
  if i != -1:
    j=s.find(']')
    th.text=th.text+s[i+1:j]+' '
    th.finishSpan()
  else:
    handleURL(L)
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(L)           # endObject must not be handled
  th.enterText(1)
  ts.restoreStyle()
  ObjContext.unset()
  # URL

def handleURL(L):
  "URLs can be very long without spaces, so split them up 'gracefully'"
  u=L[2]                # include tag
  tl=textLength(u)
  pw=gv.pageWidth*1000
  ts.setStyle('ttype')
  if tl <= pw:
    th.text=u
    return
  while tl > pw:
    i=70
    z=0
    while z<pw:
      z=textLength(u[:i])
      i=i+1
    s=u[:i]
    i=s.rfind(urlSep)
    th.text=th.text+s[:i]
    th.finishSpan()
    th.finishPara()
    u=u[i:]
    tl=textLength(u)
  if u:
    th.text=u
  # handleURL

def parseParam(s):
  "parse next parameter"
  "return number of left lines, start of next parameter, and alignment"
  i=0
  align=''
  while s[i:] and s[i] == '|':
    i=i+1                               # set i to number of left lines
  j=i
  if s[i:]:                             # if anything left
    if s[j] == 'p' or s[j] == '>':      # then if 'p' or '>'
      j=s.find('m')                     # then find 'm' to skip 'c' in 'cm'
    else:
      j=i+1                             # else set j to next char
    while s[j:] and not s[j] in 'lcr|': # find next param
      j=j+1
  if j>i:                               # if aligment char found
    align=s[i:j]                        # then return it
  return i,s[j:],align
  # parseParam

def Table(L):
  "a table float"
  "is not floated, but put in a paragraph-anchored frame to keep table and"
  "caption together"
  ObjContext.set('table')
  Len=len(gv.contentBody)       # note current length of content body
  z=gv.eqnZI+1
  gv.eqnZI=z                    # this must precede that of the graphic
  gv.lineLen=0
  extraLines=0
  gv.capLen=0
  w=0
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    handleObject(L)             # Tabular handled here, and endObject is also handled
    if gv.lineLen:
      w=max(gv.lineLen,w)
      gv.lineLen=0
      extraLines=extraLines+1
  temp=[]
  while len(gv.contentBody)>Len:
    temp.append(gv.contentBody.pop())
  gv.frameN=gv.frameN+1
  w=max(gv.tableWid,gv.capLen,w)
  h=gv.rowLines*2.5
  if gv.maxColWid>0:            # is actually max height for sideways headers
    h=h+gv.maxColWid
  if gv.capLen:
    h=h+1
  h=h+extraLines
  s=textPara+ts.makeStyleName('P')+'">'
  s=s+'<draw:text-box draw:style-name="fr3" draw:name="frame'+str(gv.frameN)+'" '
  s=s+'text:anchor-type="paragraph" '
  s=s+'svg:width="'+r(w)+'cm" svg:height="-1cm" '       # -1 causes OO to calc frame size
  s=s+'draw:z-index="'+str(z)+'">'
  gv.contentBody.append(s)
  while temp:
    gv.contentBody.append(temp.pop())
  gv.contentBody.append('</draw:text-box></text:p>')
  ObjContext.unset()
# Table

def setDefStyles(p):
  "p = e.g. {|l||c|p{3cm}|c|r|} n"
  "where n is the number of rows"
  "other cell formats are"
  ">{\centering}p{3cm} means column is centered with 3cm width"
  ">{\raggedleft}p{3cm} assume means right aligned to give ragged left border"
  ">{\raggedright}p{3cm} assume means left aligned to give ragged right border"
  gv.colWids={}
  gv.cellRow=[]
  gv.widthOveride=[]
  gv.hlineOveride=[]
  gv.cellStyles={}
  gv.Cnames=[]
  tn=str(gv.tableNo)+'.'
  j=p.rfind('}')
  s=p[1:j]                      # extract content between {}
  n=0                           # number of cells in a row
  L=[0,0,1]
  while L[2]:
    cellType=''
    L=parseParam(s)
    s=L[1]                      # remaining params
    i=L[0]                      # number of left border lines
    if L[2]:                    # if alignment char exists
      k=0
      if L[2][0] == '>':                #then if alignment given
        if L[2][3] == 'c':                #then set alignment :-
          cellType='c'
        elif L[2][9] == 'l':
          cellType='r'
        else:
          cellType='l'
        k=L[2].find('}')+1
      if L[2][k] == 'p':                # then if is size over-ride
        x=parseInteger(L[2][k:])
        gv.widthOveride.append(x)       # then set width over-ride
        if cellType == '':
          cellType='c'                  # & centre it by default
      else:
        cellType=L[2]           # else get alignment (c,l or r)
        gv.widthOveride.append(0)       # & append null over-ride
      if i == 1:
        cellType=cellType+'LS'
      elif i == 2:
        cellType=cellType+'LD'
      gv.cellRow.append(tn+cellType)
      gv.colWids[n]=[]
      gv.hlineOveride.append(0)
      n=n+1
    else:                       # when finished
      if i == 1:
        s='RS'
      elif i == 2:
        s='RD'
      s=gv.cellRow.pop()+s      # add right border to last cell in row
      gv.cellRow.append(s)
      s=''
  gv.tableCols=n
  gv.tableRows=eval(p[j+1:])
  gv.tableRow=1
  gv.sideWays=0
  i=0
  while i<n:
    gv.cellStyles[i]={}         # create null entries for cell column styles
    gv.Cnames.append(0)         # create null entries for cell name numbers
    i=i+1
  # setDefStyles

def initCell():
  "initialise variables before parsing next table cell"
  #th.finishPara()
  gv.cellStart=len(gv.contentBody)
  gv.mergecols=0
  gv.lineLen=0
  th.text=''
  # initCell

def calcTableWid():
  "calculate table width in cm"
  "a multicolumn is indicated by a negative width for a cell, and widths of"
  " -100 in following merged cells"
  # first scan for negative widths indication multicolumns -----------------------
  cw=[]
  i=0
  while i in gv.colWids:
    cw.append([])
    j=0
    while gv.colWids[i][j:]:
      if gv.colWids[i][j]<0:            # if multi-column
        cw[i].append(-gv.colWids[i][j]) # stored widths of multicolumns  in cw
        gv.colWids[i][j]=0              # and set corresponding col widths to 0
      else:
        cw[i].append(0)
      j=j+1
    i=i+1
  gv.maxColWid=0
  if gv.isSideways:                     # if header row is sideways
    i=0
    while i in gv.colWids:
      if gv.colWids[i][0]>gv.maxColWid:
        gv.maxColWid=gv.colWids[i][0]   # for figure float
      gv.colWids[i][0]=0.4              # then set all header cell widths to 0.8
      i=i+1
  i=0
  gv.colmax=[]                          # initialise list of max col widths
  while i in gv.colWids:                # calculate max width of each column
    if gv.widthOveride[i]:
      gv.colmax.append(gv.widthOveride[i])
    else:
      gv.colmax.append(round(max(gv.colWids[i])+0.4,3))
    i=i+1
  j=0
  if cw:                                # if multicolumns exist
    i=0
    while cw[i][j:]:                    # then scan cw
      while cw[i:]:
       if cw[i][j]:                     # if is multi
         k=0
         L=0
         ii=i
         w=cw[i][j]+0.3                 # then w = its width + margin
         while cw[i:] and cw[i][j]:             # scan following columns in row
           k=k+1                                # k becomes number of merged cells
           L=L+gv.colmax[i]             # and find their summed width
           i=i+1
         if L<w:                        # if is less than that of multi
           L=(w-L)/k                    # then L = increment to each merged column
           while cw[ii][j]:
             gv.colmax[ii]=gv.colmax[ii]+L      # add it to all those columns
             ii=ii+1
       else:
         i=i+1
      j=j+1                             # increment row
      i=0                               # and set column index back to 0
  gv.tableWid=sum(gv.colmax)            # table width = sum of max column widths
  # calcTableWid

def finishTable():
  "write autostyles, based on cell sizes"
  calcTableWid()
  i=(gv.pageWidth-gv.tableWid)/2
  if i < 0:
    i=0
  mw=r(i)
  s='<style:style style:name="'+gv.tableName+'" style:family="table">'
  s=s+'<style:properties style:width="'+r(gv.tableWid)+'cm" '
  s=s+'fo:margin-left="'+mw+'cm" fo:margin-right="'+mw+'cm" '
  s=s+'table:align="margins" style:may-break-between-rows="false"'+xmlSep+'></style:style>'

  # now make column styles based on greatest widths --------------------------------

  i=0
  while i in gv.colWids:
    ch=UCletters[i]             # 'A' for 0 etc.
    a='<style:style style:name="'+gv.tableName+'.'+ch+'" style:family="table-column">'
    a=a+'<style:properties style:column-width="'+r(gv.colmax[i])+'cm" '
    a=a+'/></style:style>'
    i=i+1
    s=s+a
  gv.contentAutoStyles=gv.contentAutoStyles+s+gv.cellAutoStyles
  # finishTable

def initTable():
  s='<table:table table:name="'+gv.tableName+'" table:style-name="'+gv.tableName+'">'
  i=0
  while i < gv.tableCols:
    s=s+'<table:table-column table:style-name="'+gv.tableName+'.'
    s=s+UCletters[i]+'"/>'
    i=i+1
                                        # Z is used for multicolumns
  gv.contentBody.append(s)              # initialise content body text
  gv.cellAutoStyles=''
  gv.TableRow='<table:table-row>'       # initialise current row, assembled by handleElSize()
  # initTable

def Tabular(L):
  "a fixed-position table unless is floated in a Table"
  "the control of operations allows for the sequence of entries in the content body"
  "  and autostyles to differ from that in which objects are recovered for handling"
  "  this especially affects calculation of table size which can only be done at the end"
  "cell content is accumulated in th.text"
  "the style of a cell is constructed by DOendCell, which also writes the xml-wrapped"
  "  content into gv.TableRow"
  "all rows are constructed by DOtabularnewline"
  "writeTableRow enters the row in gv.contentBody"
  "we adopt the cell border policy of specifying left and top lines as standard, so"
  "that the number of \\hlines prior to a row can be accommodated without the need"
  "  to look ahead"
  gv.tabN=gv.tabN+1
  ObjContext.set('table')               # set context to control other procedures
  gv.lineLen=0
  gv.crp=0
  gv.isSideways=0
  gv.rowLines=0
  ts.removeMargin()             # in case is nested in a list
  th.currentPara=''
  gv.tableNo=gv.tableNo+1
  gv.tableName='Table'+str(gv.tableNo)
  L=gv.extractObj()                     # get parameters
  setDefStyles(L[2])                    # set default column border styles
  gv.tableStart=len(gv.contentBody)     # start of table entries in gv.contentBody
  initTable()                           # initialise main contents entry
  initCell()
  ps='<text:p '
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    elif L[0] == gv.StartObP:   # handle compound object for next cell
      ObjContext.unset()        # remove 'table' context
      obc=gv.contentBody        # and save current body
      gv.contentBody=[]         # and ensure is null for new object
      th.currentPara=''         # ensure text para is null
      gv.eqnWidth=0             # ensure is 0
      gv.lineLen=0
      handleObject(L)           # handle the object
      s=''
      while gv.contentBody: # if result is in content body
        s=s+th.extractSpan()
      th.text=th.text+s                 # set th.text ready for end-of-cell processing
      gv.lineLen=gv.eqnWidth+gv.lineLen
      gv.contentBody=obc        # restore content body
      ObjContext.set('table')   # and 'table' context
    else:
      handleObject(L)           # handle all simple objects prior to endObject
  finishTable()                 # write autostyles
  handleObject(L)               # handle endObject
  gv.BlankInserted=0
  handleBlankLine('')
  th.paraCount=0
  ObjContext.unset()
  # Tabular

def handleEnsureMath(L):
  "enter as symbol"
  th.finishSpan()                               # close any outstanding span
  ts.saveStyle()
  ts.setFont('symbol')                          # set font for symbol
  th.text=L[2]
  th.enterText(1)                               # write completed escape sequence
  th.finishSpan()
  ts.restoreStyle()
  # handleEnsureMath

def handleFontSize(L):
  "this only changes/creates the current style, it does not write it anywhere"
  if ObjContext.state[:3] == 'eqn':
    ss=gv.fontSizeDict[L[2]]
    if ss != '}':
      x=parseInteger(ss)+ts.defFontSize
      th.text=th.text+' size '+str(x)+'{'
    else:
      th.text=th.text+'}}'
  else:
  #if ObjContext.state == 'standard':
    size=gv.fontSizeDict[L[2]]
    if size == '}':
      th.finishStyle()                  # finish Span or para & restore prev style
    else:
      ts.saveStyle()
      ts.setSize(eval(size))
    if th.textState:
      #th.enterText(1)
      th.finishSpan()

  # handleFontSize

# ISO 8859 2 - 4 dictionaries to convert to OpenOffice codes

ISO8859={
'acute':
     {'S':      [0x9a,0xc5],
      'Z':      [0xb9,0xc5],
      's':      [0x9b,0xc5],
      'z':      [0xba,0xc5],
      'R':      [0x94,0xc5],
      'L':      [0xb9,0xc4],
      'C':      [0x86,0xc4],
      'N':      [0x83,0xc5],
      'r':      [0x95,0xc5],
      'l':      [0xba,0xc4],
      'c':      [0x87,0xc4],
      'n':      [0x84,0xc5]
     },
'cedil':
       {'S':    [0x9e,0xc5],            # cedilla
        's':    [0x9f,0xc5],
        'T':    [0xa2,0xc5],
        't':    [0xa3,0xc5],
        'R':    [0x96,0xc5],
        'L':    [0xbb,0xc4],
        'G':    [0xa2,0xc4],
        'r':    [0x97,0xc5],
        'l':    [0xbc,0xc4],
        'g':    [0xa3,0xc4],
        'N':    [0x85,0xc5],
        'K':    [0xb6,0xc4],
        'n':    [0x86,0xc5],
        'k':    [0xb7,0xc4]
       },
'circ':
     {'H':      [0xa4,0xc4],            # circumflex
      'J':      [0xb4,0xc4],
      'h':      [0xa5,0xc4],
      'j':      [0xb5,0xc4],
      'C':      [0x88,0xc4],
      'G':      [0x9c,0xc4],
      'S':      [0x9c,0xc5],
      'c':      [0x89,0xc4],
      'g':      [0x94,0xc4],
      's':      [0x9d,0xc5]
     },
'hung':
    {'O':       [0x90,0xc5],            # Hungarian umlaut
     'U':       [0xb0,0xc5],
     'o':       [0x91,0xc5],
     'u':       [0xb1,0xc5]
    },
'macron':
      { 'E':    [0x92,0xc4],            # macron (i.e. overline)
        'e':    [0x93,0xc4],
        'A':    [0x80,0xc4],
        'I':    [0xaa,0xc4],
        'O':    [0x8c,0xc5],
        'U':    [0xaa,0xc5],
        'a':    [0x81,0xc4],
        'i':    [0xab,0xc4],
        'o':    [0x8d,0xc5],
        'u':    [0xab,0xc5]
      },
'tilde':
       {'I':    [0xa8,0xc4],
        'i':    [0xa9,0xc4],
        'U':    [0xa8,0xc5],
        'u':    [0xa9,0xc5]
       }
} # ISO8859

def doAccent(style):
  "needed where all accented text must be coded"
  th.finishSpan()
  L=gv.extractObj()                     # should be Text object
  c=L[2]
  s=''
  i=0
  while c[i:]:
    if c[i] == '\\':
      i=i+1
    L=ISO8859[style][c[i]]
    s=s+chr(L[1])+chr(L[0])             # translate next char
    i=i+1
  th.text=s
  # doAccent

def handleFontStyle(L):
  "this only changes/creates the current style, it does not write it anywhere"
  if ObjContext.state[:3] == 'eqn':
    th.text=th.text+gv.MathsStyleDict[L[2]]
    return
  style=gv.fontStyleDict[L[2]]
  if style == '}':
    th.finishStyle()                    # finish Span or para
    if ts.isSmallCap:
      ts.isSmallCap=0
    return
  ts.saveStyle()
  th.finishSpan()
  if ObjContext.state[:3] != 'eqn':
    s=gv.fontStyleDict[L[2]]
    if s[:2] == 'WT':                   # weight
      ts.setWeight(s[2:])
    elif s[:2] == 'ST':                 # style
      ts.setStyle(s[2:])
    elif s[:2] == 'un':                 # underline
      ts.setUnderline('single')
    elif s[:2] == 'AC':                 # macron, tilde, and Hungarian umlaut
      ts.setAccent(s[2:])               # accented chars from ISO 8859 2 - 4
                                        # all use same style
    elif s[:2] == 'CH':                 # cedil[la], circ[umflex]
      ts.setAccent(s[2:])               # more accented chars from ISO 8859 2 - 4
    elif s[:5] == 'acute' or s[:5] == 'grave':
      ts.setAccent(s[:5])               # acute chars from ISO 8859 2 - 4
    elif s[0] == '_' or s[0] == '^':
      ts.setStyle(s)
    #elif s[:2] == 'MA':                # maths blackboard, caligraphic or fraktur
    #  pass                             # ignore as cannot handle
  # handleFontStyle

"""
def ParBox(L):
  "it's function is not clear, and it only seems to occur in footnotes; ignored"
  gv.extractObj()               # skip parameters
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    handleObject(L)
  # ParBox
"""

def Array(L):
  "parse array"
  ObjContext.set('eqna')
  th.text=th.text+' matrix{'
  sl=len(th.text)
  params=gv.extractObj()[2]
  i=params.find('{')
  j=params.find('}')
  gv.arrCols=len(params[i+1:j])
  gv.arrCol=0
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(L)
  i = -2
  while th.text[sl:]:
    i=1
    while i<gv.arrCols+1:
      if params[i] != 'c':
        th.text=th.text[:sl]+'align'+params[i]+' '+th.text[sl:]
      j=th.text[sl:].find('#')
      if j != -1:
        sl=j+sl+1
      else:
        sl=10000
      i=i+1
    sl=sl+1
  th.text.rstrip('#')                   # strip redundant '#'
  th.text=th.text+'}'   # finish matrix{...} & leave text for equation to handle
  restoreContext()
  ObjContext.unset()
  # Array

def Fraction(L):
  "place {} round numerator"
  gv.lineLen=0
  th.text=th.text+'{{'          # enclose whole fraction in {} as well as numerator
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(L)
  th.text=th.text+'} over {'    # leave text for equation to handle
  restoreContext()
  # Fraction

def FracDenom(L):
  "place } at end of denominator"
  gv.lineLen=0
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(L)
  th.text=th.text+'}}'          # leave text for equation to handle
  restoreContext()
  # FracDenom

def handleBlankLine(L):
  "handle blank line"
  if ObjContext.state == 'table':
    return
  if not gv.BlankInserted:                              # if not just inserted
    th.finishPara()
    gv.contentBody.append('<text:p text:style-name="'+ts.makeStyleName('P')+'"/>')
    gv.BlankInserted=1                                  # set flag
  # handleBlankLine

def handleUnrecog(L):
  "copy verbatim for now; preferably should be fixed first"
  th.text=th.text+L[2]
  # handleUnrecog

def handleNull(L):
  "i.e. ignore item"
  return
  # handleNull

def handleEqnLabel(L):
  gv.EqnLab=L[2]
  gv.labelWidth=ts.fontScale*textLength(gv.EqnLab)
  gv.labFntSize=ts.getSize()          # needed to calculate number of padding spaces
  gv.isLabelled=1
  # handleEqnLabel

def handleErr(stateNow,newState,i,j,k):
  "repair error in syntax"
  t=th.text[j:k]
  if gv.lastParsed[0] in mustQuote:
    if stateNow in ['lb','AT','LB','SS','TX']:
      j=gv.lastParsed[1]
      k=gv.lastParsed[2]
      th.text=th.text[:j]+' ital{"'+th.text[j:k]+'"}'+th.text[k:]
      if not newState:
        return
      return 'TX',i+9
  if stateNow == 'KW':
    th.text=th.text[:j]+' ital{"'+t+'"}'+th.text[k:]
    return newState,i+9
  elif stateNow == 'MS':
    n=j
    while not th.text[n:n+5] in ['matri','stack']:
      n=n-1
    if th.text[n] == 'm':
      d=6
    else:
      d=5
    th.text=th.text[:n]+' ital{"'+th.text[n:n+d]+'"}'+th.text[n+d:]
    return newState,i+9
  elif stateNow in ['lb','AT','BO','FN','LB','RB','SS','US','UO','#']:
    th.text=th.text[:j]+'{}'+th.text[j:]
    return newState,i+2
  # handleErr

def parseRealNum(i):
  "parse real number"
  #while th.text[i:] and th.text[i].isspace():
  #  i=i+1
  j=i
  if th.text[i] in '+-':
    i=i+1
  d=0                                           # number of actual digits
  while th.text[i:] and th.text[i].isdigit():
    d=d+1
    i=i+1
  if th.text[i:]:
    if th.text[i] == '.':
      i=i+1
      while th.text[i:] and th.text[i].isdigit():
        d=d+1
        i=i+1
    if th.text[i] in 'eE':
      k=i
      i=i+1
      if th.text[i:] and th.text[i] in '+-':
        i=i+1
      expOK=d
      while th.text[i:] and th.text[i].isdigit():
        d=d+1
        i=i+1
      if expOK == d:
        i=k
  if d:
    return j,i
  return -1,0
  # parseRealNum

def parseTerm(i):
  "parse out a symbol, returning j,k.i such that symbol = s[j:k]"
  "and s[i] is first following non-whitespace"
  "'right ]' and 'left lbrace' for example are treated as single symbols"
  while th.text[i:] and th.text[i].isspace():
    i=i+1
  if th.text[i:] and th.text[i] in '()[]{}#':
    return i,i+1,i+1
  L=parseRealNum(i)
  j=L[0]
  if j != -1:
    i=L[1]
  elif th.text[i].isalnum():
    j=i
    while th.text[i:] and th.text[i].isalnum():
      i=i+1
  elif th.text[i] in ops:
    j=i
    if th.text[i] == '&':
      while th.text[i] == '&':
        while th.text[i] != ';':
          i=i+1
        i=i+1
      if th.text[i] in ops:
        i=i+1
    else:
      i=i+1
      while th.text[i:] and th.text[j:i] in Ops:
        i=i+1
      if i-j>1:
        i=i-1
  elif th.text[i] == '%':
    j=i
    i=i+1
    while th.text[i:] and th.text[i].isalpha():
      i=i+1
  else:         # else other single char
    j=i
    i=i+1
  k=i
  while th.text[i:] and th.text[i].isspace():
   i=i+1
  if th.text[j:k] in ['right','left']:
    L=parseTerm(i)
    s=th.text[L[0]:L[1]]
    if th.text[j:k] == 'left' and s in leftBR or th.text[j:k] == 'right' and s in rightBR:
      j=L[0]
      k=L[1]
      i=L[2]
  gv.lastParsed=th.text[j:k],j,k,i
  return j,k,i
  # parseTerm

"For Lyx we must:-"
"convert _ and ^ to csub and csup if eqn is not inline"
"convert sqrt[]{} to nroot{}{}"

"Starmath requires:-"
"1. Must not be followed by right bracket:-"
"       attribute font function operator size subscript superscript #"
"2. Must be followed by {"
"       matrix stack"
"3. Brackets must match"
"4. Must be in quotes if orphaned:-"
"       binom bold font from ital left matrix over right serif size stack to"
"5. size must be followed by a number"
"6. font must be followed by:- ital, italic, bold or serif"
"7. No empty matrix elements (however this is hard and is handled by handleEsc())"

"states are:-"
"['lb','AT','BO','FN','KW','LB','MS','NU','RB','SS','SY','TX','UO','US','#']"
"lb     {"
"AT     attributes"
"BO     binary operator"
"FN     function"
"KW     font,size"
"LB     left bracket"
"MS     matrix or stack"
"NU     numeric i.e. a [signed] integer"
"RB     right bracket"
"SS     subscript or superscript"
"SY     symbol"
"TX     text"
"US     unary operator special i.e. sum prod or lim"
"UO     other unary operator"
"#      matrix delimiter"

"The syntax table below is for what will not cause an error in StarMath"
"rather than for what is mathematically valid"

exceptions={ # these are error situations e.g. 'lb' must not be followed by 'BO'
        'lb':   ['#','BO','SS'],
        'AT':   ['#','RB','SS'],
        'BO':   ['#','BO','RB','SS'],
        'FN':   ['#','RB'],
        'KW':   ['#','RB','SS'],
        'LB':   ['#','BO','SS'],
        'MS':   ['AT','BO','FN','KW','LB','MS','NU','RB','SS','SY','TX','UO','US','#'],
        'SS':   ['#','RB'],
        'UO':   ['#','BO','RB','SS'],
        'US':   ['#','BO','RB'],
        '#':    ['BO','RB','SS'],
        'NU':   [],
        'RB':   [],
        'SY':   [],
        'TX':   []
}

def findMatch(s,i):
  "find matching ']'"
  more=1
  while s[i:] and more:
    if s[i] == '[':
      more=more+1
    elif s[i] == ']':
      more=more-1
    i=i+1
  return i              # if not found return i past end so that s[i:] = ''
  # findMatch

def setState(t):
  "determine state"
  if t == '{':                  # we distinguish this for the sake of 'MS'
    return 'lb'
  elif t in leftBR:
    return 'LB'
  elif t in rightBR:
    return 'RB'
  elif t in unop[:3]:
    return 'US'
  elif t in unop:
    return 'UO'
  elif t in binop:
    return 'BO'
  elif t in functions:
    return 'FN'
  elif t.isdigit() or t[0] in '+-' and t[1:].isdigit():
    return 'NU'
  elif t in symbols or t in mathsSymb:
    return 'SY'
  elif t in subsup:
    return 'SS'
  elif t in attributes:
    return 'AT'
  elif t in ['size','font']:
    return 'KW'
  elif t in ['matrix','stack']:
    return 'MS'
  elif t == '#':
    return '#'
  else:
    return 'TX'
  # setState

def mathSyntax(i,lb):
  "check and correct maths syntax"
  "assume we enter after left bracket and return after right bracket"
  ii=i
  wasSS=False
  if lb == '{':
    state='lb'
  else:
    state='LB'
  rb=matchingBR[lb]
  w=len(lb)                             # w = length of left bracket
  i=i+w
  while th.text[i:] and th.text[i].isspace():
    i=i+1
  while th.text[i:]:
    if th.text[i] == '"':
      i=th.text[i+1:].find('"')+i+2
      while th.text[i:] and th.text[i].isspace():
        i=i+1
      state='TX'
      continue
    L=parseTerm(i)
    j=L[0]
    k=L[1]
    i=L[2]
    t=th.text[j:k]
    ts=setState(t)
    if t in leftBR:                             # if left bracket
      i=mathSyntax(j,t)                         # then recurse
      state='RB'                                        # then set state
      if i<0:                                           # if brackets matched
        i = -i                                          # else set i>0, +2
    elif ts in exceptions[state]:
      L=handleErr(state,ts,i,j,k)
      state=L[0]
      i=L[1]
    elif t == rb:                               # if matching right bracket
      return i                                  # then return
    elif t in rightBR:                          # if unmatched bracket
      s=th.text[:ii]+leftBR[lb]+th.text[ii+w:j]
      i=len(s)-1                                # do not "consume" the bracket
                                                # which belongs to the caller
      th.text=s+th.text[j:]                             # then replace left bracket
      return -i                                         # & return error
    else:                                       # if syntax ok
      if ts == 'TX' and t in mustQuote:
        th.text=th.text[:j]+' ital{"'+t+'"}'+th.text[k:]
        i=i+9
        state=ts
      elif ts == 'KW':                                  # then check special cases
        L=parseTerm(i)
        i=L[2]
        s=th.text[L[0]:L[1]]
        if t == 'size' and setState(s) == 'NU':
          pass
        elif s in ['ital','italic','bold','serif']:
          pass
        else:
          th.text=th.text[:j]+' ital{"'+t+'"}'+th.text[k:]
          state='TX'
          i=i+9
      elif ts == 'SS' and (state == 'US' or wasSS) and ObjContext.state != 'eqninline':
        th.text=th.text[:j]+setCentral[t]+th.text[k:]
        wasSS=not wasSS
        i=i+4
      elif t == 'sqrt' and th.text[i] == '[':           # if is n'th root
        b=findMatch(th.text,i+1)
        if th.text[b :]:
          th.text=th.text[:b-1]+'}'+th.text[b:]         # then convert matching ']'
        else:
          pass  # what do we do here?  assume does not arise!
        th.text=th.text[:j]+'nroot{'+th.text[k+1:]      # and replace sqrt with nroot
        state=ts
      else:
        state=ts
      if not state in ['SS','US']:
        wasSS=False
  return -i
  # mathSyntax

def fixQuotes():
  "replace ` and ~ within quotes"
  s=th.text.replace('  ',' ')
  i=s.find('"')
  j=0
  while i != -1:
    i=i+j+1
    while th.text[i] != '"':
      if s[i] == '`':
        s=s[:i].rstrip()+' '+s[i+1:].lstrip()
      elif s[i] == '~':
        s=s[:i].rstrip()+'  '+s[i+1:].lstrip()
        i=i+1
      i=i+1
    j=i+1
    i=s[j:].find('"')
  th.text=s
  # fixQuotes

def checkSyntax():
  "check syntax of maths expression"
  i=0
  while th.text[i].isspace():
    i=i+1
  if i:
    th.text=th.text[i:]
  th.text='{'+th.text+'}'   # essential to ensure disjoint equation is fully checkerd
  mathSyntax(0,'{')
  th.text=th.text[1:-1]
  fixQuotes()
  # checkSyntax

def arraySize(s,i,Scale):
  "size an array"
  e=i                           # assume e[i] != '{', but array ends with '}'
  j=1
  while j:                      # find end of array
    if s[e] == '{':
      j=j+1
    elif s[e] == '}':
      j=j-1
    e=e+1
    while s[e].isspace():
      e=e+1
  m=s[i:e]                      # extract array part from s
  height=0
  hr=0
  colwids=[]                    # for max column widths
  j=m.find('#')                 # '#' delimits cells, '##' rows
  c=0
  while j != -1:
    L=calcES('{'+m[:j]+'}',0,Scale)
    if len(colwids) == c:
      colwids.append(L[0])
    else:
      colwids[c]=max(colwids[c],L[0])
    c=c+1
    hr=max(hr,L[1])
    if m[j+1] == '#':           # if end of row
      height=height+hr+rowGap   # add inter-row gap
      hr=0
      c=0
      j=j+1
    m=m[j+1:]                   # skip '#'
    j=m.find('#')               # find end of next cell
  L=calcES('{'+m+'}',0,Scale)   # handle last cell
  if len(colwids) == c:
    colwids.append(L[0])
  else:
    colwids[c]=max(colwids[c],L[0])
  hr=max(hr,L[1])
  width=sum(colwids)+colGap*(len(colwids)-1)
  height=height+hr
  return width,height,e
  # arraySize

def parseMathsTerm(s,i):
  "parse out a symbol, returning j,k.i such that symbol = s[j:k]"
  "and s[i] is first following non-whitespace"
  "'right ]' and 'left lbrace' for example are treated as single symbols"
  while s[i].isspace():
    i=i+1
  j=i
  while s[i] != '{' and s[i] != '}' and not s[i].isspace():
    i=i+1
  if s[i-1] == '_' or s[i-1] == '^':
    if s[j] != '_' and s[j] != '^':     # if first pass
      i=i-1                             # return previous term
    #else return directly
  k=i
  while s[i].isspace():
   i=i+1
  return j,k,i
  # parseMathsTerm

def sizeText(s):
  "size a mixed term"
  "return tuple of width and height as for an entry in mathsSizes"
  if s in mathsSizes:
    return mathsSizes[s]
  elif s in mathsSymb:
    return mathsSymb[s]
  else:
    w=textLength(s)
  return (w,526)                # return reasonably safe height (= that of aleph)
  # sizeText

def testLB(s,i):
  "test for left bracket of some kind"
  if s[i] == '{':
    return 1
  if s[i:i+4] == 'left':
    L=parseMathsTerm(s,i+4)
    if s[L[0]:L[1]] in LeftBR:
      return 1
  return 0
  # testLB

def calcES(s,i,Scale):
  "Recursive part of calcEqnSize;"
  "i is current index into string s, Scale is parent's temporary scaling factor;"
  "We have complicated the thing slightly by treating 'left' and 'right' as well as"
  "'{' and '}' as compound object delimiters, as the total width including bracket"
  "width is needed for calculating width of fractions correctly;"
  "N.B. width is additive while ht is a max value, only being additive for 'over'"
  "it is easy here to handle s only via indices which reduces garbage generation"
  L=()                          # ensure scope of L is ok
  if s[i] == '{':
    w=1                         # width of '{' is 1
    i=i+1                       # skip '{'
    bracew=0                    # initialise previous width
    braceScale=1
    endB='}'
  else: # is 'left'
    L=parseMathsTerm(s,i+4)     # parse out opening bracket
    bracew=mathsSizes[s[L[0]:L[1]]][0] # initialise width to that of enclosing brackets
    endB='right '+compl[s[L[0]]]+s[L[0]+1:L[1]] # set end bracket from initial one
        # e.g. left langle is matched by right rangle. or left [ by right ]
    w=len(endB)
    braceScale=1.15
    i=L[2]                      # set i past that bracket
  width=0                       # initialise width
  prevw=0
  ht=0                          # initialise height of this compound term
  numHt=0                       # height of numerator
  scale=1                       # scale changes for subscript, superscript or bold
  size=1
  isSubSup=0
  while s[i].isspace():
    i=i+1
  while True:
    if s[i:i+w] == endB:        # if end of compound term reached
      while s[i+w:] and s[i+w] == ' ':
        i=i+1
      return Scale*(width+prevw+bracew),braceScale*Scale*ht,i+w
                                # return scaled size
    elif testLB(s,i):           # else if start of new compound term
      L=calcES(s,i,scale)                       # then handle it
      scale=1                                   # & reset temporary scale change
      width=width+prevw                         # update width from previous term
      prevw=L[0]                                # and set prev width for latest term
      ht=max(ht,L[1])                           # and height to maximum found
      i=L[2]                                    # set i to start of next term
      numHt=L[1]
    elif s[i:i+6] == 'matrix':
      L=arraySize(s,i+7,scale)                  # skip 'matrix{'
      scale=1
      width=width+prevw                         # update width from previous term
      prevw=L[0]                                # and set prev width for latest term
      ht=max(ht,L[1])                           # and height to maximum found
      i=L[2]                                    # set i to start of next term
    else:                       # else handle term directly
      M=parseMathsTerm(s,i)             # parse new term
      t=s[M[0]:M[1]]                    # copy next term
      i=M[2]                            # start of next term, or '{' or '}'
      if t:                     # if term found
        if t in ['_','^']:
          L=calcES(s,i,scale*0.58)
          if isSubSup:                  # if two sub/sup in succession
            prevw=max(L[0],prevw)       # then take max of two
            isSubSup=0
          else:
            width=width+prevw           # else normal
            prevw=L[0]
            isSubSup=1
          ht=max(ht,L[1]+100*(1+isSubSup))
          i=L[2]
          continue
        elif t in ['csub','csup']:      # if is sub or sup over/under symbol
          L=calcES(s,i,scale*0.58)
          prevw=max(L[0],prevw)         # then set prev width to max of two
          ht=ht+L[1]                    # and add height correction
          if ObjContext.state != 'eqninline':
            ht=ht+100                   # not for inline as it already has extra space
          i=L[2]
        elif t == 'over':               # then if is fraction
          if testLB(s,i):               # then get size of denominator
            L=calcES(s,i,scale)
          else:
            L=parseMathsTerm(s,i)
            M=sizeText(s[L[0]:L[1]])
            L[0]=M[0]*scale             # ditto, setting L to width
            L[1]=M[1]                   # and height as for calcES call
          scale=1                       # scale is only for one [compound] term
          i=L[2]                        # set i to start of next term or '{' or '}'
          prevw=max(width+prevw,L[0])   # set max of numerator and denominator
          width=0                       # and leave all width in prevw
          ht=max(ht,numHt+L[1]+mathsSizes['over'][1]) # add height of denom and 'over' to that
                                          # of numerator
        elif t == 'font':       # else if font style change
          L=parseMathsTerm(s,i)         # then parse out style name
          i=L[2]                        # & ignore it
        elif t == 'size':               # if temporary size change
          L=parseMathsTerm(s,i)         # parse out differential size change
          x=eval(s[L[0]:L[1]])
          if s[L[0]-1] == '-':
            x = -x
          size=1.0*(x+ts.defFontSize)/ts.defFontSize    # and apply factor
          i=L[2]
        else:
          L=sizeText(t)         # else get its width and height tuple
          width=width+prevw             # and augment width from previous term
          prevw=L[0]*scale              # and set new previous term width
          ht=max(ht,L[1]*scale)         # and set height to maximum found
          if t in ts.newScale:          # if temporary scaling change required
            scale=ts.newScale[t]        # then set scale
          else:
            scale=1                     # else restore it to 1
      isSubSup=0
  # calcES

def calcEqnSize():
  "calculate equation width and height in cm"
  checkSyntax()                         # ensure format will work in OpenOffice
  if th.text.lstrip(' {}')[:4] != 'size':
    factor=int(ts.getSize()*ts.calcFactor())
    th.text='size '+str(factor)+' {'+th.text+'}'
    i=th.text.find('{')                 # skip the size prefix
  else:
    i=th.text.find('size')
    factor=parseInteger(th.text[i:])
    i=th.text.find('{')
  L=calcES(th.text[i:],0,1)     # calculate size in cm*1000 for 12pt
  if ts.fontScale == 0:         # arises if in footnote environment
    ts.fontScale=0.001
  if L[1] < 700:                # ensure minimum height of 700 so inline
    x=700                       # equations line up with the text
  else:
    x=L[1]
  gv.eqnWidth=0.0001*(L[0]+gv.eqnMargin)*factor #0.000083
  gv.eqnHeight=(x+gv.eqnHtmargin)*ts.fontScale
  # calcEqnSize

def finishEqn(needsStyleName):
  "common procedure to finish all types of equation"
  gv.objN=gv.objN+1                     # increment number of objects
  subDir=gv.objDir+str(gv.objN)
  os.mkdir(docDir+subDir)

  # write results to xml files

  gv.settingsEqn['ViewAreaWidth']=str(int(gv.eqnWidth*1000.0+0.5))
  gv.settingsEqn['ViewAreaHeight']=str(int(gv.eqnHeight*1000.0+0.5))
  makeContentEqn(subDir)                # make content.xml for equation
  makeSettingsEqn(subDir)               # make settings.xml for equation
  th.text=''                            # prevent equation being printed as text
  enterContents(subDir,needsStyleName)  # make entry for main content.xml
  gv.eqnZI=gv.eqnZI+1                   # increment z index
  gv.BlankInserted=0
  if needsStyleName:
    handleBlankLine('')                 # insert blank line
    restoreContext()
  th.paraCount=0
  gv.textF=1.0
  ObjContext.unset()
  # finishEqn

def Equation(L):
  "special handling for centred equation"
  "look for matching EndObject to finish"
  ObjContext.set('eqn')         # set context to control other procedures
  gv.isLabelled=0
  gv.lineLen=0
  gv.textF=eqnTextF
  th.text = ''
  more=1
  while more:
    nextObj=gv.extractObj()     # extract next object
    if nextObj[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(nextObj)
  calcEqnSize()
  finishEqn(1)
  # Equation

def EqInline(L):
  "special handling for inline equation"
  "look for matching EndObject to finish"
  ObjContext.set('eqninline')           # set context to control other procedures
  gv.isLabelled=0
  gv.lineLen=0
  gv.textF=eqnTextF
  if th.text:
    th.finishSpan()
  th.text=''
  more=1
  while more:
    L=gv.extractObj()           # extract next object
    if L[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(L)
  calcEqnSize()
  finishEqn(0)
  # EqInline

def addLabels(labels):
  "add labels to equation array"
  "two extra columns are appended to the array, the first to pad the thing holding"
  "phantom w's, and the second to hold the labels"
  lines=0
  mw=0                          # initialise max label width
  j=0
  while labels[j:]:
    if not 'phantom' in labels[j]:
      mw=max(textLength(labels[j]),mw)
    lines=lines+1
    j=j+1
  padding=(gv.pageWidth-gv.eqnWidth)/2-mw*ts.fontScale
  ww=gv.charWid[ord('w')]*ts.defFontSize/12.0
  n=int(padding*1000/ww+0.5)    # n = number of w's to centre the equation
                                # for right justification
  padding=''
  i=n
  while i>0:
    padding=padding+'w'
    i=i-1
  j=0
  i=0
  while lines:
    lines=lines-1
    if lines:
      i=th.text[i:].find('##')+i
      th.text=th.text[:i]+'# phantom w#'+labels[j]+th.text[i:]
      i=i+14+len(labels[j])             # allow for '# phantom W#', label and '##'
    else:
      th.text=th.text[:-2]+'# phantom {'+padding+'}#'+labels[j]+'}}'
    j=j+1
  #gv.eqnWidth=gv.eqnWidth+(mw+n*ww+100)*0.001
  gv.eqnWidth=gv.eqnWidth+ts.fontScale*(mw+n*ww+100)
  # addLabels

def EqnArray(L):
  "special handling for equation array"
  "we format the equations as an array with the left column right-aligned"
  "and the right column left-aligned"
  "if the centre column is composed of '=' signs as is most likely, we enclose"
  "the equal sign in double-quotes as StarMath goes awry if a cell starts with an"
  "equal sign"
  "if the array has any labels they are handled by adding extra array columns"
  ObjContext.set('eqn')         # set context to control other procedures
  th.text=''
  gv.isLabelled=0
  label='phantom a'
  labels=[]
  labelled=0
  gv.textF=eqnTextF
  eqn=''
  term=''
  cell=3
  more=1
  while more:
    L=gv.extractObj()
    if L[2] == 'endCell':       # if endCell
      if cell == 3:
        s='alignr '
        eqn=eqn+'alignr '+term
      elif cell == 2:
        if term.strip() in mathsOrphans+binOps:
          eqn=eqn+phTerm+term+phTerm
        else:
          eqn=eqn+term
      eqn=eqn+'#'
      cell=cell-1
      term=''
    elif L[2] == '\\newline' or L[0] == gv.EndObP:      # if newline or end of array
      eqn=eqn+'alignl '+term
      if L[2] == '\\newline':
        eqn=eqn+'##'
      labels.append(label)
      label='phantom a'
      cell=3
      term=''
    elif L[0] == gv.EqnArrP:    # if text
      term=term+L[2]            # then build up term
    elif L[0] == gv.EqnLabP:    # else if is equation label
      label=L[2]                # then set label
      labelled=1                # and set flag
    else:                       # else handle object
      handleObject(L)
      term=term+' '+th.text     # and add result to equation
      th.text=''
    if L[0] == gv.EndObP:       # if end of stack
      th.text='size '+str(ts.defFontSize)+' {matrix{'+eqn+'}}'
      more=more-1
  #ts.fontScale=ts.defFontSize/12000.0
  ts.fontScale=0.001            # otherwise we lose the labels
  calcEqnSize()
  if labelled:
    addLabels(labels)
  finishEqn(1+labelled)
  # EqnArray

def Footnote(L):
  "special handling for footnote"
  "it is messy as we have to fiddle with the span versus para tags to get the"
  "citation number embedded in the surrounding text; this arises as a footnote is"
  "necessarily an object, which starts by wrapping up any outstanding paragraph"
  "(c.f. handleStart())"
  ObjContext.set('footnote')
  gv.contentBody.pop()          # remove automatic blank line for start of new object
  s=gv.contentBody.pop()        # recover previous line
  i=s.rfind('</text:p>')        # remove possible end of para
  if i == -1:
    i=10000
  gv.contentBody.append(s[:i])  # and restore previous line
  s='<text:footnote text:id="ftn'+str(gv.ftn)+'"><text:footnote-citation>'
  s=s+str(gv.ftn+1)+'</text:footnote-citation><text:footnote-body>'
  th.text=th.text+s+'<text:p text:style-name="Footnote">'
  gv.contentBody.append(th.text) # insert text directly to avoid start span or para
  th.text=''
  th.contSpan()                 # continue span without incurring a new para
  ts.fontScale=7/12000.0
  gv.ftn=gv.ftn+1               # increment footnote number
  more=1
  while more:                   # handle rest of Footnote object
    L=gv.extractObj()
    if L[0] == gv.EndObP:
      more=more-1
    else:
      handleObject(L)
  th.finishSpan()
  th.text='</text:p></text:footnote-body></text:footnote>'
  th.addText()          # write text without incurring extra spans etc.
  restoreContext()
  ObjContext.unset()
  # Footnote

def handleMarginNote(L):
  "special handling for margin note"
  "these do not now arise as they are converted to footnotes"
  return
  # handleMarginNote

def MultiCol(L):
  "special handling for multi-column"
  "it causes adjacent columns to be merged and the content to be in the result"
  "the first param such as {2} defines the number of columns to be merged"
  "the second such as {c|} says how to format the result"
  "it is also used for special formatting of a single cell"
  ObjContext.set('multi')
  more=1
  while more:
    L=gv.extractObj()
    if L[0] == gv.ObjP:
      i=L[2].find('}')
      gv.mergecols=eval(L[2][1:i])      # extract number of columns for cell processing
      i=L[2].rfind('{')
      j=L[2].rfind('}')
      gv.mergeformat=L[2][i:j+1]        # append format
    elif L[0] == gv.EndObP:
        more=more-1
    handleObject(L)                     # handle endObject
  ObjContext.unset()
  # MultiCol

def handleParams(L):
  "object parameters are e.g. {|c|c|}, {|c|r|l|} or {MMM}"
  "c means centre the text, r means right-align it and l means left-align it"
  "they are handled by their parent objects, so any call here means they"
  "are redundant"
  return
  # handleParams

def handleVspace(L):
  "handle special vertical spacing items"
  "the item contains a vertical measurement such as 0.3cm"
  th.finishPara()
  th.enterVspace(L[2])          # N.B. the current style is not affected
  # handleVspace

def Sideways(L):
  "special handling for string printed vertically"
  """
  this is specified in styles.xml for the style 'Table Heading' with e.g
   style:text-rotation-angle="90", which means that the first row of the table
  must be specified with <table:table-header-rows> ... </table:table-header-rows>
  Only tables requiring sideways text should use header rows as the specification
  automatically applies to all tables
  we cannot solve the problem for individual cells as OpenOffice ignores the rotation
  style property if specified for them; hence we only act on the first sideways
  object and then only if it is in the first row
  """
  if gv.tableRow == 1:
    gv.sideWays=1               # set flag for writeTableRow()
    gv.needsHeaderRow=1         # set flag for styles.xml file
  handleObjBody(L)
  # Sideways

def handleObject(L):
  "special handling for objects: calls function listed in dictionary"
  try:
    exec('s='+specFunc[L[0]][1]+'(L)')
  except:
    s=None      # udefined object name
  return s
  # handleObject

def parseDict(s):
  "parse dictionary with first line in s"
  "the entry after each key may be an implied tuple, an explicit tuple,"
  "a list or a subordinate dictionary;"
  "if comments exist in the file then an implied tuple must have a comma after"
  "the last term, even for the last entry of the dictionary;"
  "keys that are ascii numbers are converted to integers;"
  "similarly for entry items"
  d={}
  while True:
    if d or not s:
      try:
        s=gv.ipf.readline().strip()
      except:
        return d
    if not s or s[0] == '}':
      return d
    i=s.find(':')
    if i == -1:
      return None
    ss=s[:i].strip()
    try:
      key=eval(ss)
    except:
      key=ss
    s=s[i+1:].strip()
    j=1
    if s[0] == '(':
      e=s.rfind(')')
    elif s[0] == '[':
      e=s.rfind(']')
    elif s[0] == '{':
      L=parseDict(s[1:])
      if not L:
        return None
      d[key]=L
      continue
    else:
      e=10000
      j=0
    s=s[j:e].strip()
    t=[]
    i=s.find(',')
    while i != -1 or s:
      if i == -1:
        i=s.find('#')           # strip off any comment at end of line
        if i == -1:
          i=10000
        ss=s[:i].strip()
        i=10000
      else:
        ss=s[:i].strip()
      try:
        x=eval(ss)
        t.append(x)
      except:
        if ss:
          t.append(ss)
      s=s[i+1:]
      i=s.find(',')
    if len(t) == 1:
      t=t[0]
    d[key]=t
  # parseDict

def readDict(dict):
  "read a dictionary-formatted ascii file"
  d={}
  entry=()
  try:
    gv.ipf=open(dicPath+dict,'rt')
  except:
    makeErrorFile(noOpen+dicPath+dict)
    return {}
  s=gv.ipf.readline()
  d=parseDict(s)
  gv.ipf.close()
  if not d:
    makeErrorFile(noOpen+dicPath+dict)
    reportErr()
    sys.exit(0)
  return d
  # readDict

def readAsciiDict(dict):
  "this enables us to construct a dictionary from a simple ascii file"
  "the odd lines are the keys and the ensuing even lines the corresponding entries"
  d={}
  ok=1
  try:
    ff=open(dicPath+dict,'rt')
  except:
    makeErrorFile(noOpen+dicPath+dict)
    return {}
  while ok:
    s=ff.readline()
    entry=ff.readline()
    if not s or not entry:
      ff.close()
      return d
    d[s[:-1]]=entry[:-1]                # remove '\n' in each case
  # readAsciiDict

def extractStyle(pstyle,s,n):
  i=pstyle.find(s)
  if i == -1:
    return i
  L=parseInt(pstyle[i:])
  factor=1.0
  s=L[1]
  if s == 'mm':
    factor=0.1
  elif s == 'in':
    factor=2.54
  x=L[0]*factor
  if n == 1:
    ts.TopMargin=r(x)
  elif n == 2:
    ts.BottomMargin=r(x)
  elif n == 3:
    ts.LeftMargin=r(x)
  elif n == 4:
    ts.RightMargin=r(x)
  elif n == 5:
    ts.PageHeight=r(x)
  return x
  # extractStyle

def setPageStyle(pstyle):
  "set page style parameters"
  extractStyle(pstyle,'tmargin',1)
  extractStyle(pstyle,'bmargin',2)
  lm=extractStyle(pstyle,'lmargin',3)
  rm=extractStyle(pstyle,'rmargin',4)
  if 'landscape' in pstyle:
    ts.orientation='landscape'
  pw=eval(ts.PageWidth)
  if 'paperwidth' in pstyle:
    pw=extractStyle(pstyle,'paperwidth',0)
    extractStyle(pstyle,'paperheight',5)
  elif 'a5' in pstyle:
    pw=14.8
    ts.PageHeight='21'
  elif 'a3' in pstyle:
    pw=29.7
    ts.PageHeight='42'
  elif 'b3' in pstyle:
    pw=35.3
    ts.PageHeight='50'
  elif 'b4' in pstyle:
    pw=25
    ts.PageHeight='35.3'
  elif 'b5' in pstyle:
    pw=17.6
    ts.PageHeight='25'
  elif 'US letter' in pstyle:
    pw=21.59
    ts.PageHeight='27.94'
  elif 'US legal' in pstyle:
    pw=21.59
    PageHeight='35.56'
  elif 'US executive' in pstyle:
    pw=18.41
    ts.PageHeight='26.67'
  ts.PageWidth=r(pw)
  if rm == -1:
    rm=eval(ts.RightMargin)
  if lm == -1:
    lm=eval(ts.LeftMargin)
  gv.pageWidth=pw-rm-lm
  if gv.pageWidth <= 0:
    makeErrorFile('Margins too big')
    print 'Margins too big'
    sys.exit(2)
  i=pstyle.find('pt,')
  if i != -1:
    ts.defFontSize=eval(pstyle[i-2:i])
  i=pstyle.find('line-spacing')
  if i != -1:
    j=pstyle[i:].find(',')+i
    ts.lineSpacing='fo:line-height="'+pstyle[i+13:j]+'%" '
  i=pstyle.find('twoside')
  if i != -1 and lm != rm:
    ts.mirror='style:page-usage="mirrored"'
  i=pstyle.find('font=')
  if i != -1:
    j=pstyle[i:].find(',')+i
    ts.defFont=pstyle[i+5:j]
    fn='<style:font-decl style:name="'+ts.defFont+'" fo:font-family="&apos;'
    fn=fn+ts.defFont+'&apos;" '
    ts.defFontDecl.append(fn)
    fn=''
    if ts.defFont[:5] in ['Times','Palati','New C']:
      fn='roman'
    elif ts.defFont[:4] == 'Helv':
      fn='swiss'
    if fn:
      ts.defFontDecl.append('style:font-family-generic="'+fn+'" ')
    else:
      ts.defFontDecl.append('')
    ts.defFontDecl.append('style:font-pitch="variable"/>')
  # setPageStyle

def convDoc():
  """ -----------------------------------------------------------------------------
        first read required dictionaries
  """
  gv.fontSizeDict=readAsciiDict('FontSize'+dicType)
  if not gv.fontSizeDict:
    return
  gv.fontStyleDict=readAsciiDict('FontStyle'+dicType)
  if not gv.fontSizeDict:
    return
  gv.MathsStyleDict=readAsciiDict('FontStyleMaths'+dicType)
  if not gv.fontSizeDict:
    return
  gv.escapeDict=readAsciiDict('Esc'+dicType)
  if not gv.escapeDict:
    return

  """ ----------------------------------------------------------------------------
        open files
  """

  try:
    ff=open(document,'rt')
  except:
    makeErrorFile(noOpen+document)
    return

  """ ------------------------------------------------------------------------------
        unpickle main object dictionary to set up objDict,
        set various index variables

  """

  s=ff.readline()
  if s[:8] == 'Geometry':               # if non-default page style
    s=pickle.load(ff)                   # then read params
    setPageStyle(s)                     # and parse them out
    s=ff.readline()                     # object name
  gv.objDict=pickle.load(ff)            # load object dictionary
  gv.setIndices()                       # set object index variables
  gv.setHspaces()
  obj=gv.findObj('')
  ff.close()

  """ -----------------------------------------------------------------------------
        read the structure list
  """

# STARTS HERE

  gv.resetIndices()
  gv.BlankInserted=0
  ts.initStyle()
  ObjContext.set('standard')    # set context to control other procedures
  saveContext(1)
  while True:
    L=gv.extractObj()      # extract next object; this terminates loop when all done
    if L[0] == gv.TextP:
      th.text=th.text+L[2]
      th.enterText(1)
    else:
      handleObject(L)
  # convDoc

#------------------------------------------------------------------------------

# preliminary processing :-

def makeDocDir():
  "empty old diry and then make new"
  try:
    os.mkdir(docDir)            # if success
    return                      # dir did not exist
  except:
    pass                        # else empty it
  os.chdir(docDir)
  files=os.listdir(docDir)
  for f in files:
    try:
      files1=os.listdir(f)
      os.chdir(f)
      for fs in files1:
        os.remove(fs)
      os.chdir(docDir)
      os.rmdir(f)
    except:
      os.remove(f)
  # makeDocDir

# MAIN PROGRAM --------------------------------------------------------------------

#  read in presets (paths and printer setup)

gv.ipf=open('ConvTexConfig')
homeDir=gv.ipf.readline().strip()               # home or base director
s=gv.ipf.readline().strip()                     # target dir
picDir=gv.ipf.readline().strip()                # default path for pictures
printerSetup=gv.ipf.readline().strip()          # printer setup string
gv.ipf.close()

#  read in paths to suit platform

gv.ipf=open('ConvTexPaths')                     # open presets file in current dir
Tmp=gv.ipf.readline().strip()                   # temp path
ErrPath=gv.ipf.readline().strip()               # path for error file
docDir=gv.ipf.readline().strip()                # where xml document will be assembled
gv.ipf.readline().strip()                       # skip development path
dicPath=homeDir+gv.ipf.readline().strip()       # path for dictionaries
dirySep=gv.ipf.readline().strip()               # directory separator path
xmlSep=gv.ipf.readline().strip()                # XML separator
urlSep=gv.ipf.readline().strip()                # url path separator
gv.ipf.close()
xmlDir=dicPath+dirySep+'OfficeXML'+dirySep      # path for template xml files
i=s[:-1].rfind(dirySep)
gv.targetDir=s[:i+1]                            # where final file will be created

args = getopt.getopt(sys.argv[1:],'dummy')      # get run-time arguments

try:
  document=sys.argv[1]                          # if none
except:
  print fnr                                     #'File name required'
  makeErrorFile(fnr)
  sys.exit(2)
try:
  s=sys.argv[2]                         # if called by client program
except:
  isTerminal=1
  gv.docname=document
  document=Tmp+dirySep+document+'.ftd'

makeErrorFile('Execution error')                # contingency error file
gv.syntaxErr=1                                  # for syntax or exec error

#----------------------------------------------------------------------------------

makeDocDir()                                    # make new document directory

#i=sys.argv[0].rfind(dirySep)                   # find program file name
#dicType=sys.argv[0][i+5:-3]                    # extract document type from it
dicType='Office'                                # better for Windows version

#----------------------------------------------------------------------------------
#  read in various global dictionaries

charCount=readDict('charCount')                 # for calculating char widths
calcCharWid()                                   # calculate char widths
nonAscii=readDict('nonAscii')                   # non-ascii char widths
mathsSizes=readDict('mathsSizes')               # maths keyword char sizes
ts.newScale['bold']=mathsSizes['bold'][0]       # set factor for bold type
     # N.B. StarMath function func e^{} is omitted as we do not generate it from Lyx
mathsSymb=readDict('mathsSymb')                 # maths sybmol char sizes

#----------------------------------------------------------------------------------

make_xRef()                     #make cross ref from names in specFunc to its indices
convDoc()                                       # process document
reportErr()                                     # is error if we exit here
